// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024,2025 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/bytearray.h>
#include <mobius/crypt/cipher.h>
#include <mobius/crypt/hash.h>
#include <mobius/crypt/hmac.h>
#include <mobius/crypt/pkcs5.h>
#include <mobius/crypt/rot13.h>
#include <mobius/unittest.h>
#include <cstring>

#include <iostream>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/TDES_Core.pdf
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
testcase_cipher_3des ()
{
  mobius::unittest test ("mobius::crypt::cipher -> 3des");

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Constants
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray tdes_plaintext =
  {
    0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
    0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
  };

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // 3DES - mode ECB
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const std::string plaintext0 = "6bc1bee22e409f96e93d7e117393172aae2d8a571e03ac9c9eb76fac45af8e51";

  auto des0 = mobius::crypt::new_cipher_ecb ("3des", {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x01, 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef});

  mobius::bytearray b0;
  b0.from_hexstring (plaintext0);

  mobius::bytearray b1 = des0.encrypt (b0);
  test.ASSERT_EQUAL (b1.to_hexstring (), "06ede3d82884090aff322c19f0518486730576972a666e58b6c88cf107340d3d");

  b1 = des0.decrypt (b1);
  test.ASSERT_EQUAL (b1.to_hexstring (), plaintext0);

  // 3DES - mode ECB
  auto des1 = mobius::crypt::new_cipher_ecb ("3des", {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x01, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x01, 0x23});

  b1 = des1.encrypt (b0);
  test.ASSERT_EQUAL (b1.to_hexstring (), "714772f339841d34267fcc4bd2949cc3ee11c22a576a303876183f99c0b6de87");

  b1 = des1.decrypt (b1);
  test.ASSERT_EQUAL (b1.to_hexstring (), plaintext0);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/TDES_CBC.pdf
  // CBC-TDES (1)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto cm = mobius::crypt::new_cipher_cbc ("3des",
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x01, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x01, 0x23},
    {0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17}
  );

  const mobius::bytearray tdes_ciphertext_1 =
  {
    0x20, 0x79, 0xc3, 0xd5, 0x3a, 0xa7, 0x63, 0xe1, 0x93, 0xb7, 0x9e, 0x25, 0x69, 0xab, 0x52, 0x62,
    0x51, 0x65, 0x70, 0x48, 0x1f, 0x25, 0xb5, 0x0f, 0x73, 0xc0, 0xbd, 0xa8, 0x5c, 0x8e, 0x0d, 0xa7,
  };

  auto b = cm.encrypt (tdes_plaintext);
  test.ASSERT_EQUAL (b, tdes_ciphertext_1);

  cm.reset ();
  b = cm.decrypt (tdes_ciphertext_1);
  test.ASSERT_EQUAL (b, tdes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/TDES_CBC.pdf
  // CBC-TDES (2)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  cm = mobius::crypt::new_cipher_cbc ("3des",
    {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
     0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0x01,
     0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
    {0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17}
  );

  const mobius::bytearray tdes_ciphertext_2 =
  {
    0x74, 0x01, 0xce, 0x1e, 0xab, 0x6d, 0x00, 0x3c, 0xaf, 0xf8, 0x4b, 0xf4, 0x7b, 0x36, 0xcc, 0x21,
    0x54, 0xf0, 0x23, 0x8f, 0x9f, 0xfe, 0xcd, 0x8f, 0x6a, 0xcf, 0x11, 0x83, 0x92, 0xb4, 0x55, 0x81,
  };

  b = cm.encrypt (tdes_plaintext);
  test.ASSERT_EQUAL (b, tdes_ciphertext_2);

  cm.reset ();
  b = cm.decrypt (tdes_ciphertext_2);
  test.ASSERT_EQUAL (b, tdes_plaintext);

  test.end ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Test cipher aes
//! \see https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
//! \see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/AES_ModesA_All.pdf
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
testcase_cipher_aes ()
{
  mobius::unittest test ("mobius::crypt::cipher -> aes");

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Constants
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_iv =
  {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
  };

  const mobius::bytearray aes_plaintext =
  {
    0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
    0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
    0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
    0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
  };

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES128.Encrypt (F.1.1) and AES128.Decrypt (F.1.2)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ecb_1 =
  {
    0x3a, 0xd7, 0x7b, 0xb4, 0x0d, 0x7a, 0x36, 0x60, 0xa8, 0x9e, 0xca, 0xf3, 0x24, 0x66, 0xef, 0x97,
    0xf5, 0xd3, 0xd5, 0x85, 0x03, 0xb9, 0x69, 0x9d, 0xe7, 0x85, 0x89, 0x5a, 0x96, 0xfd, 0xba, 0xaf,
    0x43, 0xb1, 0xcd, 0x7f, 0x59, 0x8e, 0xce, 0x23, 0x88, 0x1b, 0x00, 0xe3, 0xed, 0x03, 0x06, 0x88,
    0x7b, 0x0c, 0x78, 0x5e, 0x27, 0xe8, 0xad, 0x3f, 0x82, 0x23, 0x20, 0x71, 0x04, 0x72, 0x5d, 0xd4
  };

  auto cm = mobius::crypt::new_cipher_ecb (
       "aes",
       {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c}
  );

  auto b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ecb_1);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ecb_1);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES192.Encrypt (F.1.3) and AES192.Decrypt (F.1.4)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ecb_2 =
  {
    0xbd, 0x33, 0x4f, 0x1d, 0x6e, 0x45, 0xf2, 0x5f, 0xf7, 0x12, 0xa2, 0x14, 0x57, 0x1f, 0xa5, 0xcc,
    0x97, 0x41, 0x04, 0x84, 0x6d, 0x0a, 0xd3, 0xad, 0x77, 0x34, 0xec, 0xb3, 0xec, 0xee, 0x4e, 0xef,
    0xef, 0x7a, 0xfd, 0x22, 0x70, 0xe2, 0xe6, 0x0a, 0xdc, 0xe0, 0xba, 0x2f, 0xac, 0xe6, 0x44, 0x4e,
    0x9a, 0x4b, 0x41, 0xba, 0x73, 0x8d, 0x6c, 0x72, 0xfb, 0x16, 0x69, 0x16, 0x03, 0xc1, 0x8e, 0x0e
  };

  cm = mobius::crypt::new_cipher_ecb (
       "aes",
       {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b}
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ecb_2);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ecb_2);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES256.Encrypt (F.1.5) and AES192.Decrypt (F.1.6)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ecb_3 =
  {
    0xf3, 0xee, 0xd1, 0xbd, 0xb5, 0xd2, 0xa0, 0x3c, 0x06, 0x4b, 0x5a, 0x7e, 0x3d, 0xb1, 0x81, 0xf8,
    0x59, 0x1c, 0xcb, 0x10, 0xd4, 0x10, 0xed, 0x26, 0xdc, 0x5b, 0xa7, 0x4a, 0x31, 0x36, 0x28, 0x70,
    0xb6, 0xed, 0x21, 0xb9, 0x9c, 0xa6, 0xf4, 0xf9, 0xf1, 0x53, 0xe7, 0xb1, 0xbe, 0xaf, 0xed, 0x1d,
    0x23, 0x30, 0x4b, 0x7a, 0x39, 0xf9, 0xf3, 0xff, 0x06, 0x7d, 0x8d, 0x8f, 0x9e, 0x24, 0xec, 0xc7
  };

  cm = mobius::crypt::new_cipher_ecb ("aes",
       {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4}
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ecb_3);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ecb_3);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES128.Encrypt (F.2.1) and AES128.Decrypt (F.2.2)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_1 =
  {
    0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46, 0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d,
    0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee, 0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2,
    0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b, 0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16,
    0x3f, 0xf1, 0xca, 0xa1, 0x68, 0x1f, 0xac, 0x09, 0x12, 0x0e, 0xca, 0x30, 0x75, 0x86, 0xe1, 0xa7
  };

  cm = mobius::crypt::new_cipher_cbc (
       "aes",
       {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
       aes_iv
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_1);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_1);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES192.Encrypt (F.2.3) and AES192.Decrypt (F.2.4)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_2 =
  {
    0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d, 0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8,
    0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4, 0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a,
    0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0, 0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0,
    0x08, 0xb0, 0xe2, 0x79, 0x88, 0x59, 0x88, 0x81, 0xd9, 0x20, 0xa9, 0xe6, 0x4f, 0x56, 0x15, 0xcd,
  };

  cm = mobius::crypt::new_cipher_cbc (
       "aes",
       {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
       aes_iv
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_2);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_2);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES256.Encrypt (F.2.5) and AES192.Decrypt (F.2.6)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_3 =
  {
    0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6,
    0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d, 0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d,
    0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf, 0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61,
    0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc, 0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b,
  };

  cm = mobius::crypt::new_cipher_cbc ("aes",
         {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
         aes_iv
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_3);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_3);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES128.Encrypt (F.4.1) and AES128.Decrypt (F.4.2)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ofb_1 =
  {
    0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20, 0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a,
    0x77, 0x89, 0x50, 0x8d, 0x16, 0x91, 0x8f, 0x03, 0xf5, 0x3c, 0x52, 0xda, 0xc5, 0x4e, 0xd8, 0x25,
    0x97, 0x40, 0x05, 0x1e, 0x9c, 0x5f, 0xec, 0xf6, 0x43, 0x44, 0xf7, 0xa8, 0x22, 0x60, 0xed, 0xcc,
    0x30, 0x4c, 0x65, 0x28, 0xf6, 0x59, 0xc7, 0x78, 0x66, 0xa5, 0x10, 0xd9, 0xc1, 0xd6, 0xae, 0x5e,
  };

  cm = mobius::crypt::new_cipher_ofb (
       "aes",
       {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
       aes_iv
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ofb_1);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ofb_1);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES192.Encrypt (F.4.3) and AES192.Decrypt (F.4.4)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ofb_2 =
  {
    0xcd, 0xc8, 0x0d, 0x6f, 0xdd, 0xf1, 0x8c, 0xab, 0x34, 0xc2, 0x59, 0x09, 0xc9, 0x9a, 0x41, 0x74,
    0xfc, 0xc2, 0x8b, 0x8d, 0x4c, 0x63, 0x83, 0x7c, 0x09, 0xe8, 0x17, 0x00, 0xc1, 0x10, 0x04, 0x01,
    0x8d, 0x9a, 0x9a, 0xea, 0xc0, 0xf6, 0x59, 0x6f, 0x55, 0x9c, 0x6d, 0x4d, 0xaf, 0x59, 0xa5, 0xf2,
    0x6d, 0x9f, 0x20, 0x08, 0x57, 0xca, 0x6c, 0x3e, 0x9c, 0xac, 0x52, 0x4b, 0xd9, 0xac, 0xc9, 0x2a,
  };

  cm = mobius::crypt::new_cipher_ofb (
       "aes",
       {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
       aes_iv
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ofb_2);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ofb_2);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES256.Encrypt (F.4.5) and AES192.Decrypt (F.4.6)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ofb_3 =
  {
    0xdc, 0x7e, 0x84, 0xbf, 0xda, 0x79, 0x16, 0x4b, 0x7e, 0xcd, 0x84, 0x86, 0x98, 0x5d, 0x38, 0x60,
    0x4f, 0xeb, 0xdc, 0x67, 0x40, 0xd2, 0x0b, 0x3a, 0xc8, 0x8f, 0x6a, 0xd8, 0x2a, 0x4f, 0xb0, 0x8d,
    0x71, 0xab, 0x47, 0xa0, 0x86, 0xe8, 0x6e, 0xed, 0xf3, 0x9d, 0x1c, 0x5b, 0xba, 0x97, 0xc4, 0x08,
    0x01, 0x26, 0x14, 0x1d, 0x67, 0xf3, 0x7b, 0xe8, 0x53, 0x8f, 0x5a, 0x8b, 0xe7, 0x40, 0xe4, 0x84,
  };

  cm = mobius::crypt::new_cipher_ofb ("aes",
         {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
         aes_iv
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ofb_3);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ofb_3);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES128.Encrypt (F.5.1) and AES128.Decrypt (F.5.2)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ctr_1 =
  {
    0x87, 0x4d, 0x61, 0x91, 0xb6, 0x20, 0xe3, 0x26, 0x1b, 0xef, 0x68, 0x64, 0x99, 0x0d, 0xb6, 0xce,
    0x98, 0x06, 0xf6, 0x6b, 0x79, 0x70, 0xfd, 0xff, 0x86, 0x17, 0x18, 0x7b, 0xb9, 0xff, 0xfd, 0xff,
    0x5a, 0xe4, 0xdf, 0x3e, 0xdb, 0xd5, 0xd3, 0x5e, 0x5b, 0x4f, 0x09, 0x02, 0x0d, 0xb0, 0x3e, 0xab,
    0x1e, 0x03, 0x1d, 0xda, 0x2f, 0xbe, 0x03, 0xd1, 0x79, 0x21, 0x70, 0xa0, 0xf3, 0x00, 0x9c, 0xee,
  };

  cm = mobius::crypt::new_cipher_ctr (
       "aes",
       {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
       {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff}
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ctr_1);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ctr_1);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES192.Encrypt (F.5.3) and AES192.Decrypt (F.5.4)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ctr_2 =
  {
    0x1a, 0xbc, 0x93, 0x24, 0x17, 0x52, 0x1c, 0xa2, 0x4f, 0x2b, 0x04, 0x59, 0xfe, 0x7e, 0x6e, 0x0b,
    0x09, 0x03, 0x39, 0xec, 0x0a, 0xa6, 0xfa, 0xef, 0xd5, 0xcc, 0xc2, 0xc6, 0xf4, 0xce, 0x8e, 0x94,
    0x1e, 0x36, 0xb2, 0x6b, 0xd1, 0xeb, 0xc6, 0x70, 0xd1, 0xbd, 0x1d, 0x66, 0x56, 0x20, 0xab, 0xf7,
    0x4f, 0x78, 0xa7, 0xf6, 0xd2, 0x98, 0x09, 0x58, 0x5a, 0x97, 0xda, 0xec, 0x58, 0xc6, 0xb0, 0x50,
  };

  cm = mobius::crypt::new_cipher_ctr (
       "aes",
       {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
       {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff}
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ctr_2);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ctr_2);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
  // AES256.Encrypt (F.5.5) and AES192.Decrypt (F.5.6)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray aes_ciphertext_ctr_3 =
  {
    0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5, 0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
    0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a, 0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
    0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c, 0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d,
    0xdf, 0xc9, 0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6, 0x13, 0xc2, 0xdd, 0x08, 0x45, 0x79, 0x41, 0xa6,
  };

  cm = mobius::crypt::new_cipher_ctr ("aes",
       {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
       {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff}
  );

  b = cm.encrypt (aes_plaintext);
  test.ASSERT_EQUAL (b, aes_ciphertext_ctr_3);

  cm.reset ();
  b = cm.decrypt (aes_ciphertext_ctr_3);
  test.ASSERT_EQUAL (b, aes_plaintext);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // IGE-AES128 - test vector 1
  //! \see https://mgp25.com/blog/2015/06/21/AESIGE/
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray plaintext_ige_1 (32);  // 32 * {0}

  const mobius::bytearray ciphertext_ige_1 =
  {
     0x1a, 0x85, 0x19, 0xa6, 0x55, 0x7b, 0xe6, 0x52,
     0xe9, 0xda, 0x8e, 0x43, 0xda, 0x4e, 0xf4, 0x45,
     0x3c, 0xf4, 0x56, 0xb4, 0xca, 0x48, 0x8a, 0xa3,
     0x83, 0xc7, 0x9c, 0x98, 0xb3, 0x47, 0x97, 0xcb
  };

  cm = mobius::crypt::new_cipher_ige (
    "aes",
    {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x7,
     0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
    },
    {
     0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
     0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
     0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
     0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f
    }
  );

  test.ASSERT_EQUAL (cm.encrypt (plaintext_ige_1), ciphertext_ige_1);
  cm.reset ();
  test.ASSERT_EQUAL (cm.decrypt (ciphertext_ige_1), plaintext_ige_1);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // IGE-AES128 - test vector 2
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray plaintext_ige_2 =
  {
     0x99, 0x70, 0x64, 0x87, 0xa1, 0xcd, 0xe6, 0x13,
     0xbc, 0x6d, 0xe0, 0xb6, 0xf2, 0x4b, 0x1c, 0x7a,
     0xa4, 0x48, 0xc8, 0xb9, 0xc3, 0x40, 0x3e, 0x34,
     0x67, 0xa8, 0xca, 0xd8, 0x93, 0x40, 0xf5, 0x3b
  };

  const mobius::bytearray ciphertext_ige_2 =
  {
     0x4c, 0x2e, 0x20, 0x4c, 0x65, 0x74, 0x27, 0x73,
     0x20, 0x68, 0x6f, 0x70, 0x65, 0x20, 0x42, 0x65,
     0x6e, 0x20, 0x67, 0x6f, 0x74, 0x20, 0x69, 0x74,
     0x20, 0x72, 0x69, 0x67, 0x68, 0x74, 0x21, 0x0a
  };

  cm = mobius::crypt::new_cipher_ige (
    "aes",
    {0x54, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20,
     0x61, 0x6e, 0x20, 0x69, 0x6d, 0x70, 0x6c, 0x65
    },
    {
     0x6d, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f,
     0x6e, 0x20, 0x6f, 0x66, 0x20, 0x49, 0x47, 0x45,
     0x20, 0x6d, 0x6f, 0x64, 0x65, 0x20, 0x66, 0x6f,
     0x72, 0x20, 0x4f, 0x70, 0x65, 0x6e, 0x53, 0x53
    }
  );

  test.ASSERT_EQUAL (cm.encrypt (plaintext_ige_2), ciphertext_ige_2);
  cm.reset ();
  test.ASSERT_EQUAL (cm.decrypt (ciphertext_ige_2), plaintext_ige_2);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // AES CBC-CTS
  //! \see Libgcrypt tests/basic.c
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  cm = mobius::crypt::new_cipher_cbc_cts (
       "aes",
       {0x63,0x68,0x69,0x63,0x6b,0x65,0x6e,0x20,0x74,0x65,0x72,0x69,0x79,0x61,0x6b,0x69}
  );

  const mobius::bytearray plaintext_cbc_cts = "I would like the General Gau's Chicken, please, and wonton soup.";

  const mobius::bytearray ciphertext_cbs_cts[] = {
     {0xc6,0x35,0x35,0x68,0xf2,0xbf,0x8c,0xb4,0xd8,0xa5,0x80,0x36,0x2d,0xa7,0xff,0x7f,0x97},

     {0xfc,0x00,0x78,0x3e,0x0e,0xfd,0xb2,0xc1,0xd4,0x45,0xd4,0xc8,0xef,0xf7,0xed,0x22,
      0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0,0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5},

     {0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5,0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8,
      0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0,0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84},

     {0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0,0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
      0xb3,0xff,0xfd,0x94,0x0c,0x16,0xa1,0x8c,0x1b,0x55,0x49,0xd2,0xf8,0x38,0x02,0x9e,
      0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5,0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5},

     {0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0,0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
      0x9d,0xad,0x8b,0xbb,0x96,0xc4,0xcd,0xc0,0x3b,0xc1,0x03,0xe1,0xa1,0x94,0xbb,0xd8,
     0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5,0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8},

     {0x97,0x68,0x72,0x68,0xd6,0xec,0xcc,0xc0,0xc0,0x7b,0x25,0xe2,0x5e,0xcf,0xe5,0x84,
      0x39,0x31,0x25,0x23,0xa7,0x86,0x62,0xd5,0xbe,0x7f,0xcb,0xcc,0x98,0xeb,0xf5,0xa8,
      0x48,0x07,0xef,0xe8,0x36,0xee,0x89,0xa5,0x26,0x73,0x0d,0xbc,0x2f,0x7b,0xc8,0x40,
     0x9d,0xad,0x8b,0xbb,0x96,0xc4,0xcd,0xc0,0x3b,0xc1,0x03,0xe1,0xa1,0x94,0xbb,0xd8},
  };

  for (int i = 0;i < 6;i++)
    {
      auto ciphertext = ciphertext_cbs_cts[i];
      auto plaintext = plaintext_cbc_cts.slice (0, ciphertext.size () - 1);

      cm.reset ();
      test.ASSERT_EQUAL (cm.encrypt (plaintext), ciphertext);

      cm.reset ();
      test.ASSERT_EQUAL (cm.decrypt (ciphertext), plaintext);
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // AES GCM
  //! \see Libgcrypt tests/basic.c
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  static const struct tv
  {
    mobius::bytearray key;
    mobius::bytearray iv;
    mobius::bytearray aad;
    mobius::bytearray plaintext;
    mobius::bytearray ciphertext;
    mobius::bytearray tag;
  } tv[32] =
  {
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {
        0x58,0xe2,0xfc,0xce,0xfa,0x7e,0x30,0x61,0x36,0x7f,0x1d,0x57,0xa4,0xe7,0x45,
      },
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {
        0x58,0xe2,0xfc,0xce,0xfa,0x7e,0x30,0x61,0x36,0x7f,0x1d,0x57,0xa4,0xe7,
      },
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {
        0x58,0xe2,0xfc,0xce,0xfa,0x7e,0x30,0x61,0x36,0x7f,0x1d,0x57,0xa4,
      },
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {
        0x58,0xe2,0xfc,0xce,0xfa,0x7e,0x30,0x61,0x36,0x7f,0x1d,0x57,
      },
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {
        0x58,0xe2,0xfc,0xce,0xfa,0x7e,0x30,0x61,0x36,0x7f,0x1d,
      },
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {
        0x58,0xe2,0xfc,0xce,0xfa,0x7e,0x30,0x61,
      },
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {
        0x58,0xe2,0xfc,0xce,
      },
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {},
      {},
      {
        0x58,
      },
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x03,0x88,0xda,0xce,0x60,0xb6,0xa3,0x92,0xf3,0x28,0xc2,0xb9,0x71,0xb2,0xfe,0x78,
      },
      {},
    },
    {
      {
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
      },
      {
        0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,
      },
      {},
      {
        0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
        0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
        0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
        0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,0x1a,0xaf,0xd2,0x55,
      },
      {
        0x42,0x83,0x1e,0xc2,0x21,0x77,0x74,0x24,0x4b,0x72,0x21,0xb7,0x84,0xd0,0xd4,0x9c,
        0xe3,0xaa,0x21,0x2f,0x2c,0x02,0xa4,0xe0,0x35,0xc1,0x7e,0x23,0x29,0xac,0xa1,0x2e,
        0x21,0xd5,0x14,0xb2,0x54,0x66,0x93,0x1c,0x7d,0x8f,0x6a,0x5a,0xac,0x84,0xaa,0x05,
        0x1b,0xa3,0x0b,0x39,0x6a,0x0a,0xac,0x97,0x3d,0x58,0xe0,0x91,0x47,0x3f,0x59,0x85,
      },
      {},
    },
    {
      {
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
      },
      {
        0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,
      },
      {
        0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
        0xab,0xad,0xda,0xd2,
      },
      {
        0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
        0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
        0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
        0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,
      },
      {
        0x42,0x83,0x1e,0xc2,0x21,0x77,0x74,0x24,0x4b,0x72,0x21,0xb7,0x84,0xd0,0xd4,0x9c,
        0xe3,0xaa,0x21,0x2f,0x2c,0x02,0xa4,0xe0,0x35,0xc1,0x7e,0x23,0x29,0xac,0xa1,0x2e,
        0x21,0xd5,0x14,0xb2,0x54,0x66,0x93,0x1c,0x7d,0x8f,0x6a,0x5a,0xac,0x84,0xaa,0x05,
        0x1b,0xa3,0x0b,0x39,0x6a,0x0a,0xac,0x97,0x3d,0x58,0xe0,0x91,
      },
      {},
    },
    {
      {
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
      },
      {
        0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,
      },
      {
        0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
        0xab,0xad,0xda,0xd2,
      },
      {
        0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
        0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
        0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
        0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,
      },
      {
        0x61,0x35,0x3b,0x4c,0x28,0x06,0x93,0x4a,0x77,0x7f,0xf5,0x1f,0xa2,0x2a,0x47,0x55,
        0x69,0x9b,0x2a,0x71,0x4f,0xcd,0xc6,0xf8,0x37,0x66,0xe5,0xf9,0x7b,0x6c,0x74,0x23,
        0x73,0x80,0x69,0x00,0xe4,0x9f,0x24,0xb2,0x2b,0x09,0x75,0x44,0xd4,0x89,0x6b,0x42,
        0x49,0x89,0xb5,0xe1,0xeb,0xac,0x0f,0x07,0xc2,0x3f,0x45,0x98,
      },
      {},
    },
    {
      {
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
      },
      {
        0x93,0x13,0x22,0x5d,0xf8,0x84,0x06,0xe5,0x55,0x90,0x9c,0x5a,0xff,0x52,0x69,0xaa,
        0x6a,0x7a,0x95,0x38,0x53,0x4f,0x7d,0xa1,0xe4,0xc3,0x03,0xd2,0xa3,0x18,0xa7,0x28,
        0xc3,0xc0,0xc9,0x51,0x56,0x80,0x95,0x39,0xfc,0xf0,0xe2,0x42,0x9a,0x6b,0x52,0x54,
        0x16,0xae,0xdb,0xf5,0xa0,0xde,0x6a,0x57,0xa6,0x37,0xb3,0x9b,
      },
      {
        0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
        0xab,0xad,0xda,0xd2,
      },
      {
        0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
        0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
        0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
        0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,
      },
      {
        0x8c,0xe2,0x49,0x98,0x62,0x56,0x15,0xb6,0x03,0xa0,0x33,0xac,0xa1,0x3f,0xb8,0x94,
        0xbe,0x91,0x12,0xa5,0xc3,0xa2,0x11,0xa8,0xba,0x26,0x2a,0x3c,0xca,0x7e,0x2c,0xa7,
        0x01,0xe4,0xa9,0xa4,0xfb,0xa4,0x3c,0x90,0xcc,0xdc,0xb2,0x81,0xd4,0x8c,0x7c,0x6f,
        0xd6,0x28,0x75,0xd2,0xac,0xa4,0x17,0x03,0x4c,0x34,0xae,0xe5,
      },
      {},
    },
    {
      {
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,
      },
      {
        0x93,0x13,0x22,0x5d,0xf8,0x84,0x06,0xe5,0x55,0x90,0x9c,0x5a,0xff,0x52,0x69,0xaa,
        0x6a,0x7a,0x95,0x38,0x53,0x4f,0x7d,0xa1,0xe4,0xc3,0x03,0xd2,0xa3,0x18,0xa7,0x28,
        0xc3,0xc0,0xc9,0x51,0x56,0x80,0x95,0x39,0xfc,0xf0,0xe2,0x42,0x9a,0x6b,0x52,0x54,
        0x16,0xae,0xdb,0xf5,0xa0,0xde,0x6a,0x57,0xa6,0x37,0xb3,0x9b,
      },
      {
        0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
        0xab,0xad,0xda,0xd2,
      },
      {
        0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
        0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
        0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
        0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,
      },
      {
        0xd2,0x7e,0x88,0x68,0x1c,0xe3,0x24,0x3c,0x48,0x30,0x16,0x5a,0x8f,0xdc,0xf9,0xff,
        0x1d,0xe9,0xa1,0xd8,0xe6,0xb4,0x47,0xef,0x6e,0xf7,0xb7,0x98,0x28,0x66,0x6e,0x45,
        0x81,0xe7,0x90,0x12,0xaf,0x34,0xdd,0xd9,0xe2,0xf0,0x37,0x58,0x9b,0x29,0x2d,0xb3,
        0xe6,0x7c,0x03,0x67,0x45,0xfa,0x22,0xe7,0xe9,0xb7,0x37,0x3b,
      },
      {},
    },
    {
      {
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
      },
      {
        0x93,0x13,0x22,0x5d,0xf8,0x84,0x06,0xe5,0x55,0x90,0x9c,0x5a,0xff,0x52,0x69,0xaa,
        0x6a,0x7a,0x95,0x38,0x53,0x4f,0x7d,0xa1,0xe4,0xc3,0x03,0xd2,0xa3,0x18,0xa7,0x28,
        0xc3,0xc0,0xc9,0x51,0x56,0x80,0x95,0x39,0xfc,0xf0,0xe2,0x42,0x9a,0x6b,0x52,0x54,
        0x16,0xae,0xdb,0xf5,0xa0,0xde,0x6a,0x57,0xa6,0x37,0xb3,0x9b,
      },
      {
        0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
        0xab,0xad,0xda,0xd2,
      },
      {
        0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
        0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
        0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
        0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,
      },
      {
        0x5a,0x8d,0xef,0x2f,0x0c,0x9e,0x53,0xf1,0xf7,0x5d,0x78,0x53,0x65,0x9e,0x2a,0x20,
        0xee,0xb2,0xb2,0x2a,0xaf,0xde,0x64,0x19,0xa0,0x58,0xab,0x4f,0x6f,0x74,0x6b,0xf4,
        0x0f,0xc0,0xc3,0xb7,0x80,0xf2,0x44,0x45,0x2d,0xa3,0xeb,0xf1,0xc5,0xd8,0x2c,0xde,
        0xa2,0x41,0x89,0x97,0x20,0x0e,0xf8,0x2e,0x44,0xae,0x7e,0x3f,
      },
      {},
    },
    {
      {
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
        0xfe,0xff,0xe9,0x92,0x86,0x65,0x73,0x1c,0x6d,0x6a,0x8f,0x94,0x67,0x30,0x83,0x08,
      },
      {
        0xca,0xfe,0xba,0xbe,0xfa,0xce,0xdb,0xad,0xde,0xca,0xf8,0x88,
      },
      {
        0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,0xfe,0xed,0xfa,0xce,0xde,0xad,0xbe,0xef,
        0xab,0xad,0xda,0xd2,
      },
      {
        0xd9,0x31,0x32,0x25,0xf8,0x84,0x06,0xe5,0xa5,0x59,0x09,0xc5,0xaf,0xf5,0x26,0x9a,
        0x86,0xa7,0xa9,0x53,0x15,0x34,0xf7,0xda,0x2e,0x4c,0x30,0x3d,0x8a,0x31,0x8a,0x72,
        0x1c,0x3c,0x0c,0x95,0x95,0x68,0x09,0x53,0x2f,0xcf,0x0e,0x24,0x49,0xa6,0xb5,0x25,
        0xb1,0x6a,0xed,0xf5,0xaa,0x0d,0xe6,0x57,0xba,0x63,0x7b,0x39,
      },
      {
        0x52,0x2d,0xc1,0xf0,0x99,0x56,0x7d,0x07,0xf4,0x7f,0x37,0xa3,0x2a,0x84,0x42,0x7d,
        0x64,0x3a,0x8c,0xdc,0xbf,0xe5,0xc0,0xc9,0x75,0x98,0xa2,0xbd,0x25,0x55,0xd1,0xaa,
        0x8c,0xb0,0x8e,0x48,0x59,0x0d,0xbb,0x3d,0xa7,0xb0,0x8b,0x10,0x56,0x82,0x88,0x38,
        0xc5,0xf6,0x1e,0x63,0x93,0xba,0x7a,0x0a,0xbc,0xc9,0xf6,0x62,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x86,0xdd,0x40,0xe7,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x7d,0x6e,0x38,0xfd,0xd0,0x04,0x9d,0x28,0xdf,0x4a,0x10,0x3f,0xa3,0x9e,0xf8,0xf8,
        0x6c,0x2c,0x10,0xa7,0x91,0xab,0xc0,0x86,0xd4,0x6d,0x69,0xea,0x58,0xc4,0xf9,0xc0,
        0xd4,0xee,0xc2,0xb0,0x9d,0x36,0xae,0xe7,0xc9,0xa9,0x1f,0x71,0xa8,0xee,0xa2,0x1d,
        0x20,0xfd,0x79,0xc7,0xd9,0xc4,0x90,0x51,0x38,0x97,0xb6,0x9f,0x55,0xea,0xf3,0xf0,
        0x78,0xb4,0xd3,0x8c,0xa9,0x9b,0x32,0x7d,0x19,0x36,0x96,0xbc,0x8e,0xab,0x80,0x9f,
        0x61,0x56,0xcc,0xbd,0x3a,0x80,0xc6,0x69,0x37,0x0a,0x89,0x89,0x21,0x82,0xb7,0x79,
        0x6d,0xe9,0xb4,0x34,0xc4,0x31,0xe0,0xbe,0x71,0xad,0xf3,0x50,0x05,0xb2,0x61,0xab,
        0xb3,0x1a,0x80,0x57,0xcf,0xe1,0x11,0x26,0xcb,0xa9,0xd1,0xf6,0x58,0x46,0xf1,0x69,
        0xa2,0xb8,0x42,0x3c,0xe8,0x28,0x13,0xca,0x58,0xd9,0x28,0x99,0xf8,0xc8,0x17,0x32,
        0x4a,0xf9,0xb3,0x4c,0x7a,0x47,0xad,0xe4,0x77,0x64,0xec,0x70,0xa1,0x01,0x0b,0x88,
        0xe7,0x30,0x0b,0xbd,0x66,0x25,0x39,0x1e,0x51,0x67,0xee,0xec,0xdf,0xb8,0x24,0x5d,
        0x7f,0xcb,0xee,0x7a,0x4e,0xa9,0x93,0xf0,0xa1,0x84,0x7b,0xfe,0x5a,0xe3,0x86,0xb2,
        0xfb,0xcd,0x39,0xe7,0x1e,0x5e,0x48,0x65,0x4b,0x50,0x2b,0x4a,0x99,0x46,0x3f,0x6f,
        0xdb,0xd9,0x97,0xdb,0xe5,0x6d,0xa4,0xdd,0x6c,0x18,0x64,0x5e,0xae,0x7e,0x2c,0xd3,
        0xb4,0xf3,0x57,0x5c,0xb5,0xf8,0x7f,0xe5,0x87,0xb5,0x35,0xdb,0x80,0x38,0x6e,0x2c,
        0x5c,0xdd,0xeb,0x7c,0x63,0xac,0xe4,0xb5,0x5a,0x6a,0x40,0x6d,0x72,0x69,0x9a,0xa9,
        0x8f,0x5e,0x93,0x91,0x4d,0xce,0xeb,0x87,0xf5,0x25,0xed,0x75,0x6b,0x3b,0x1a,0xf2,
        0x0c,0xd2,0xa4,0x10,0x45,0xd2,0x87,0xae,0x29,0x6d,0xeb,0xea,0x66,0x5f,0xa0,0xc2,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x8d,0xd1,0xc1,0xdf,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0xac,0x6a,0x10,0x3f,0xe2,0x8d,0xed,0x27,0x55,0x14,0xca,0x1f,0x03,0x67,0x0a,0xa8,
        0xa1,0x07,0xbf,0x00,0x73,0x5b,0x64,0xef,0xac,0x30,0x83,0x81,0x48,0x4c,0xaa,0xd5,
        0xff,0xca,0xef,0x2f,0x77,0xbe,0xfe,0x1b,0x20,0x5c,0x86,0x19,0xc7,0xf9,0x11,0x99,
        0x27,0xc5,0x57,0xa7,0x0a,0xc2,0xa8,0x05,0xd9,0x07,0x2b,0xb9,0x38,0xa4,0xef,0x58,
        0x92,0x74,0xcf,0x89,0xc7,0xba,0xfc,0xb9,0x70,0xac,0x86,0xe2,0x31,0xba,0x7c,0xf9,
        0xc4,0xe2,0xe0,0x4c,0x1b,0xe4,0x3f,0x75,0x83,0x5c,0x40,0x0e,0xa4,0x13,0x8b,0x04,
        0x60,0x78,0x57,0x29,0xbb,0xe6,0x61,0x93,0xe3,0x16,0xf9,0x58,0x07,0x75,0xd0,0x96,
        0xfb,0x8f,0x6d,0x1e,0x49,0x0f,0xd5,0x31,0x9e,0xee,0x31,0xe6,0x0a,0x85,0x93,0x49,
        0x22,0xcf,0xd6,0x1b,0x40,0x44,0x63,0x9c,0x95,0xaf,0xf0,0x44,0x23,0x51,0x37,0x92,
        0x0d,0xa0,0x22,0x37,0xb9,0x6d,0x13,0xf9,0x78,0xba,0x27,0x27,0xed,0x08,0x7e,0x35,
        0xe4,0xe2,0x28,0xeb,0x0e,0xbe,0x3d,0xce,0x89,0x93,0x35,0x84,0x0f,0xa0,0xf9,0x8d,
        0x94,0xe9,0x5a,0xec,0xd4,0x0d,0x1f,0x5c,0xbe,0x6f,0x8e,0x6a,0x4d,0x10,0x65,0xbb,
        0xc7,0x0b,0xa0,0xd5,0x5c,0x20,0x80,0x0b,0x4a,0x43,0xa6,0xe1,0xb0,0xe0,0x56,0x6a,
        0xde,0x90,0xe0,0x6a,0x45,0xe7,0xc2,0xd2,0x69,0x9b,0xc6,0x62,0x11,0xe3,0x2b,0xa5,
        0x45,0x98,0xb0,0x80,0xd3,0x57,0x4d,0x1f,0x09,0x83,0x58,0xd4,0x4d,0xa6,0xc5,0x95,
        0x87,0x59,0xb0,0x58,0x6c,0x81,0x49,0xc5,0x95,0x18,0x23,0x1b,0x6f,0x10,0x86,0xa2,
        0xd9,0x56,0x19,0x30,0xec,0xd3,0x4a,0x4b,0xe8,0x1c,0x11,0x37,0xfb,0x31,0x60,0x4d,
        0x4f,0x9b,0xc4,0x95,0xba,0xda,0x49,0x43,0x6c,0xc7,0x3d,0x5b,0x13,0xf9,0x91,0xf8,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x76,0x8c,0x18,0x92,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x3d,0x6f,0x4e,0xf6,0xd2,0x6f,0x4e,0xce,0xa6,0xb4,0x4a,0x9e,0xcb,0x57,0x13,0x90,
        0x51,0x3b,0xf6,0xb2,0x40,0x55,0x0c,0x2c,0xa2,0x85,0x44,0x72,0xf2,0x90,0xaf,0x6b,
        0x86,0x8c,0x75,0x2a,0x9c,0xd6,0x52,0x50,0xee,0xc6,0x5f,0x59,0xbc,0x8d,0x18,0xd7,
        0x87,0xa5,0x7f,0xa0,0x13,0xd1,0x5d,0x54,0x77,0x30,0xe2,0x5d,0x1b,0x4f,0x87,0x9f,
        0x3a,0x41,0xcb,0x6a,0xdf,0x44,0x4f,0xa2,0x1a,0xbc,0xfb,0x4b,0x16,0x67,0xed,0x59,
        0x65,0xf0,0x77,0x48,0xca,0xfd,0xf0,0xb6,0x90,0x65,0xca,0x23,0x09,0xca,0x83,0x43,
        0x8f,0xf0,0x78,0xb4,0x5f,0x96,0x2a,0xfd,0x29,0xae,0xda,0x62,0x85,0xc5,0x87,0x4b,
        0x2a,0x3f,0xba,0xbe,0x15,0x5e,0xb0,0x4e,0x8e,0xe7,0x66,0xae,0xb4,0x80,0x66,0x90,
        0x10,0x9d,0x81,0xb9,0x64,0xd3,0x36,0x00,0xb2,0x95,0xa8,0x7d,0xaf,0x54,0xf8,0xbd,
        0x8f,0x7a,0xb1,0xa1,0xde,0x09,0x0d,0x10,0xc8,0x8e,0x1e,0x18,0x2c,0x1e,0x73,0x71,
        0x2f,0x1e,0xfd,0x16,0x6e,0xbe,0xe1,0x3e,0xe5,0xb4,0xb5,0xbf,0x03,0x63,0xf4,0x5a,
        0x0d,0xeb,0xff,0xe0,0x61,0x80,0x67,0x51,0xb4,0xa3,0x1f,0x18,0xa5,0xa9,0xf1,0x9a,
        0xeb,0x2a,0x7f,0x56,0xb6,0x01,0x88,0x82,0x78,0xdb,0xec,0xb7,0x92,0xfd,0xef,0x56,
        0x55,0xd3,0x72,0x35,0xcd,0xa4,0x0d,0x19,0x6a,0xb6,0x79,0x91,0xd5,0xcb,0x0e,0x3b,
        0xfb,0xea,0xa3,0x55,0x9f,0x77,0xfb,0x75,0xc2,0x3e,0x09,0x02,0x73,0x7a,0xff,0x0e,
        0xa5,0xf0,0x83,0x11,0xeb,0xe7,0xff,0x3b,0xd0,0xfd,0x7a,0x07,0x53,0x63,0x43,0x89,
        0xf5,0x7b,0xc4,0x7d,0x3b,0x2c,0x9b,0xca,0x1c,0xf6,0xb2,0xab,0x13,0xf5,0xc4,0x2a,
        0xbf,0x46,0x77,0x3b,0x09,0xdd,0xd1,0x80,0xef,0x55,0x11,0x3e,0xd8,0xe4,0x42,0x22,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x9b,0xc8,0xc3,0xaf,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x33,0x5f,0xdc,0x8d,0x5d,0x77,0x7b,0x78,0xc1,0x5b,0x7b,0xb3,0xd9,0x08,0x9a,0x0c,
        0xce,0x63,0x4e,0xef,0x19,0xf8,0x8c,0x7a,0xcb,0x31,0x39,0x93,0x69,0x7a,0x2c,0x97,
        0x3a,0xb4,0x52,0x45,0x9e,0x7b,0x78,0xbc,0xa9,0xad,0x54,0x7f,0x88,0xa6,0xae,0xd5,
        0xc0,0x8b,0x7a,0xe4,0x23,0x6b,0xb2,0x29,0x98,0xea,0x25,0x7a,0xae,0x11,0x0c,0xc9,
        0xf3,0x77,0xa1,0x74,0x82,0xde,0x0c,0xec,0x68,0xce,0x94,0xfd,0xb0,0xa0,0xc5,0x32,
        0xd6,0xbb,0xc3,0xe7,0xed,0x3c,0x6f,0x0b,0x53,0x9d,0xf3,0xc8,0xeb,0x4e,0xee,0x99,
        0x19,0xc7,0x16,0xd1,0xa5,0x59,0x1d,0xa9,0xd3,0xe6,0x43,0x52,0x74,0x61,0x28,0xe6,
        0xac,0xd8,0x47,0x63,0xc2,0xb7,0x53,0x39,0xc1,0x9a,0xb0,0xa3,0xa4,0x26,0x14,0xd0,
        0x88,0xa9,0x8c,0xc5,0x6d,0xe9,0x21,0x7c,0xb9,0xa5,0xab,0x67,0xe3,0x8d,0xe9,0x1d,
        0xe3,0x1c,0x7b,0xcd,0xa4,0x12,0x0c,0xd7,0xa6,0x5d,0x41,0xcf,0xdd,0x3d,0xfc,0xbc,
        0x2a,0xbb,0xa2,0x7a,0x9c,0x4b,0x3a,0x42,0x6c,0x98,0x1d,0x50,0x99,0x9c,0xfb,0xda,
        0x21,0x09,0x2a,0x31,0xff,0x05,0xeb,0xa5,0xf1,0xba,0x65,0x78,0xbe,0x15,0x8e,0x84,
        0x35,0xdd,0x45,0x29,0xcc,0xcd,0x32,0x2d,0x27,0xe9,0xa8,0x94,0x4b,0x16,0x16,0xcc,
        0xab,0xf2,0xec,0xfb,0xa0,0xb5,0x9d,0x39,0x81,0x3e,0xec,0x5e,0x3d,0x13,0xd1,0x83,
        0x04,0x79,0x2d,0xbb,0x2c,0x76,0x76,0x93,0x28,0x77,0x27,0x13,0xdd,0x1d,0x3e,0x89,
        0x3e,0x37,0x46,0x4c,0xb8,0x34,0xbe,0xbf,0x9f,0x4f,0x9f,0x37,0xff,0x0c,0xe6,0x14,
        0x14,0x66,0x52,0x41,0x18,0xa9,0x39,0x2b,0x0c,0xe5,0x44,0x04,0xb0,0x93,0x06,0x64,
        0x67,0xf7,0xa0,0x19,0xa7,0x61,0xcf,0x03,0x7b,0xcb,0xc8,0xb3,0x88,0x28,0xe4,0xe7,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x95,0x1a,0xe2,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0xd8,0x32,0x5a,0xe3,0x55,0x8e,0xb3,0xc2,0x51,0x84,0x2b,0x09,0x01,0x5e,0x6c,0xfb,
        0x4a,0xc4,0x88,0xa0,0x33,0xe7,0x3e,0xbf,0xe5,0x7c,0xd2,0x00,0x4c,0x1a,0x85,0x32,
        0x34,0xec,0x38,0x9d,0x18,0x5f,0xf1,0x50,0x61,0x82,0xee,0xf3,0x84,0x5a,0x84,0x4e,
        0xeb,0x29,0x08,0x4c,0x7b,0xb5,0x27,0xec,0x7d,0x79,0x77,0xd7,0xa1,0x68,0x91,0x32,
        0x2d,0xf3,0x38,0xa9,0xd6,0x27,0x16,0xfb,0x7d,0x8b,0x09,0x5e,0xcf,0x1b,0x74,0x6d,
        0xcf,0x51,0x91,0x91,0xa1,0xe7,0x40,0x19,0x43,0x7b,0x0d,0xa5,0xa9,0xa5,0xf4,0x2e,
        0x7f,0x1c,0xc7,0xba,0xa2,0xea,0x00,0xdd,0x24,0x01,0xa8,0x66,0x1e,0x88,0xf1,0xf6,
        0x0c,0x9a,0xd6,0x2b,0xda,0x3f,0x3e,0xb2,0x98,0xea,0x89,0xc7,0xc6,0x63,0x27,0xb7,
        0x6a,0x48,0x9a,0xee,0x1e,0x70,0xa0,0xc8,0xec,0x3d,0xc3,0x3e,0xb5,0xf0,0xc2,0xb1,
        0xb9,0x71,0x1a,0x69,0x9d,0xdd,0x72,0x1e,0xfe,0x72,0xa0,0x21,0xb8,0x9f,0x18,0x96,
        0x26,0xcf,0x89,0x2e,0x92,0xf1,0x02,0x65,0xa5,0xb4,0x2e,0xb7,0x4e,0x12,0xbd,0xa0,
        0x48,0xbe,0xf6,0x5c,0xef,0x7e,0xf3,0x0a,0xcf,0x9d,0x1f,0x1e,0x14,0x70,0x3e,0xa0,
        0x01,0x0f,0x14,0xbf,0x38,0x10,0x3a,0x3f,0x3f,0xc2,0x76,0xe0,0xb0,0xe0,0x7c,0xc6,
        0x77,0x6d,0x7f,0x69,0x8e,0xa0,0x4b,0x00,0xc3,0x9d,0xf9,0x0b,0x7f,0x8a,0x8e,0xd3,
        0x17,0x58,0x40,0xfe,0xaf,0xf4,0x16,0x3a,0x65,0xff,0xce,0x85,0xbb,0x80,0xfa,0xb8,
        0x34,0xc9,0xef,0x3a,0xdd,0x04,0x46,0xca,0x8f,0x70,0x48,0xbc,0x1c,0x71,0x4d,0x6a,
        0x17,0x30,0x32,0x87,0x2e,0x2e,0x54,0x9e,0x3f,0x15,0xed,0x17,0xd7,0xa1,0xcf,0x6c,
        0x5d,0x0f,0x3c,0xee,0xf5,0x96,0xf1,0x8f,0x68,0x1c,0xbc,0x27,0xdc,0x10,0x3c,0x3c,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x6b,0x99,0x9b,0xda,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x7a,0x74,0x57,0xe7,0xc1,0xb8,0x7e,0xcf,0x91,0x98,0xf4,0x1a,0xa4,0xdb,0x4d,0x2c,
        0x6e,0xdc,0x05,0x0b,0xd1,0x16,0xdf,0x25,0xa8,0x1e,0x42,0xa6,0xf9,0x09,0x36,0xfb,
        0x02,0x8a,0x10,0x7d,0xa1,0x07,0x88,0x40,0xb7,0x41,0xfd,0x64,0xf6,0xe3,0x92,0x20,
        0xfd,0xc9,0xde,0xbd,0x88,0x46,0xd3,0x1f,0x20,0x14,0x73,0x86,0x09,0xb6,0x68,0x61,
        0x64,0x90,0xda,0x24,0xa8,0x0f,0x6a,0x10,0xc5,0x01,0xbf,0x52,0x8a,0xee,0x23,0x44,
        0xd5,0xb0,0xd8,0x68,0x5e,0x77,0xc3,0x62,0xed,0xcb,0x3c,0x1b,0x0c,0x1f,0x13,0x92,
        0x2c,0x74,0x6d,0xee,0x40,0x1b,0x6b,0xfe,0xbe,0x3c,0xb8,0x02,0xdd,0x24,0x9d,0xd3,
        0x3d,0x4e,0xd3,0x9b,0x18,0xfd,0xd6,0x8f,0x95,0xef,0xa3,0xbf,0xa9,0x2f,0x33,0xa8,
        0xc2,0x37,0x69,0x58,0x92,0x42,0x3a,0x30,0x46,0x12,0x1b,0x2c,0x04,0xf0,0xbf,0xa9,
        0x79,0x55,0xcd,0xac,0x45,0x36,0x79,0xc0,0xb4,0xb2,0x5f,0x82,0x88,0x49,0xe8,0xa3,
        0xbf,0x33,0x41,0x7a,0xcb,0xc4,0x11,0x0e,0xcc,0x61,0xed,0xd1,0x6b,0x59,0x5f,0x9d,
        0x20,0x6f,0x85,0x01,0xd0,0x16,0x2a,0x51,0x1b,0x79,0x35,0x42,0x5e,0x49,0xdf,0x6f,
        0x64,0x68,0x31,0xac,0x49,0x34,0xfb,0x2b,0xbd,0xb1,0xd9,0x12,0x4e,0x4b,0x16,0xc5,
        0xa6,0xfe,0x15,0xd3,0xaf,0xac,0x51,0x08,0x95,0x1f,0x8c,0xd2,0x52,0x37,0x8b,0x88,
        0xf3,0x20,0xe2,0xf7,0x09,0x55,0x82,0x83,0x1c,0x38,0x5f,0x17,0xfc,0x37,0x26,0x21,
        0xb8,0xf1,0xfe,0xa9,0xac,0x54,0x1e,0x53,0x83,0x53,0x3f,0x43,0xe4,0x67,0x22,0xd5,
        0x86,0xec,0xf2,0xb6,0x4a,0x8b,0x8a,0x66,0xea,0xe0,0x92,0x50,0x3b,0x51,0xe4,0x00,
        0x25,0x2a,0x7a,0x64,0x14,0xd6,0x09,0xe1,0x6c,0x75,0x32,0x28,0x53,0x5e,0xb3,0xab,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x90,0xc4,0x42,0x97,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0xf5,0xc1,0xed,0xb8,0x7f,0x55,0x7b,0xb5,0x47,0xed,0xaa,0x42,0xd2,0xda,0x33,0x41,
        0x4a,0xe0,0x36,0x6d,0x51,0x28,0x40,0x9c,0x35,0xfb,0x11,0x65,0x18,0x83,0x9c,0xb5,
        0x02,0xb2,0xa7,0xe5,0x52,0x27,0xa4,0xe8,0x57,0x3d,0xb3,0xf5,0xea,0xcb,0x21,0x07,
        0x67,0xbe,0xbe,0x0f,0xf6,0xaa,0x32,0xa1,0x4b,0x5e,0x79,0x4f,0x50,0x67,0xcd,0x80,
        0xfc,0xf1,0x65,0xf2,0x6c,0xd0,0xdb,0x17,0xcc,0xf9,0x52,0x93,0xfd,0x5e,0xa6,0xb9,
        0x5c,0x9f,0xa8,0xc6,0x36,0xb7,0x80,0x80,0x6a,0xea,0x62,0xdc,0x61,0x13,0x45,0xbe,
        0xab,0x8f,0xd8,0x99,0x17,0x51,0x9b,0x29,0x04,0x6e,0xdb,0x3e,0x9f,0x83,0xc6,0x35,
        0xb3,0x90,0xce,0xcc,0x74,0xec,0xcb,0x04,0x41,0xac,0xb1,0x92,0xde,0x20,0xb1,0x67,
        0xb0,0x38,0x14,0xaa,0x7d,0xee,0x3c,0xb2,0xd3,0xbb,0x2f,0x88,0x0b,0x73,0xcf,0x7b,
        0x69,0xc1,0x55,0x5b,0x2b,0xf2,0xd4,0x38,0x2b,0x3c,0xef,0x04,0xc9,0x14,0x7c,0x31,
        0xd6,0x61,0x88,0xa8,0xb3,0x8c,0x69,0xb4,0xbc,0xaa,0x0d,0x15,0xd2,0xd5,0x27,0x63,
        0xc4,0xa4,0x80,0xe9,0x2b,0xe9,0xd2,0x34,0xc9,0x0e,0x3f,0x7b,0xd3,0x43,0x0d,0x47,
        0x5d,0x37,0x8e,0x42,0xa4,0x4e,0xef,0xcd,0xbb,0x3a,0x5b,0xa4,0xe1,0xb0,0x8d,0x64,
        0xb7,0x0b,0x58,0x52,0xec,0x55,0xd0,0xef,0x23,0xfe,0xf2,0x8d,0xe0,0xd1,0x6a,0x2c,
        0xaa,0x1c,0x03,0xc7,0x3e,0x58,0x4c,0x61,0x72,0x07,0xc6,0xfd,0x0e,0xbc,0xd4,0x6b,
        0x99,0x4f,0x91,0xda,0xff,0x6f,0xea,0x81,0x0c,0x76,0x85,0x5d,0x0c,0x7f,0x1c,0xb8,
        0x84,0x8c,0x2f,0xe1,0x36,0x3e,0x68,0xa0,0x57,0xf5,0xdf,0x13,0x0a,0xd6,0xe1,0xcd,
        0xae,0x23,0x99,0x4e,0xed,0x7a,0x72,0x1b,0x7c,0xe5,0x65,0xd1,0xb7,0xcf,0x2f,0x73,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb7,0xfa,0xc7,0x4f,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x14,0x33,0xc6,0x9d,0x04,0xd3,0x48,0x29,0x0c,0x6a,0x24,0x27,0xdf,0x5f,0x0a,0xd2,
        0x71,0xd6,0xd0,0x18,0x04,0xc0,0x9f,0x72,0x0a,0x60,0xb7,0x10,0x52,0x56,0xf7,0xae,
        0x64,0xb0,0x28,0xd4,0xfd,0x25,0x93,0x8e,0x67,0x7e,0xac,0xc2,0x93,0xc7,0x54,0x2e,
        0x82,0x93,0x88,0x6a,0xb9,0x8b,0x73,0xbc,0x88,0xec,0x27,0xdd,0x4f,0x9b,0x21,0x9e,
        0x77,0x98,0x70,0x0b,0xf4,0xd8,0x55,0xfe,0xf4,0xc3,0x3a,0xcb,0xca,0x3a,0xfb,0xd4,
        0x52,0x72,0x2f,0xf8,0xac,0xa9,0x6a,0xf5,0x13,0xab,0x7a,0x2e,0x9f,0x52,0x41,0xbd,
        0x87,0x90,0x68,0xad,0x17,0xbd,0x5a,0xff,0xc3,0xc6,0x10,0x4d,0xc1,0xfe,0xfc,0x72,
        0x21,0xb5,0x53,0x4a,0x3f,0xe0,0x15,0x9f,0x29,0x36,0x23,0xc0,0x9a,0x31,0xb2,0x0f,
        0xcd,0x2f,0xa6,0xd0,0xfc,0xe6,0x4d,0xed,0x68,0xb3,0x3d,0x26,0x67,0xab,0x40,0xf0,
        0xab,0xcf,0x72,0xc0,0x50,0xb1,0x1e,0x86,0x38,0xe2,0xe0,0x46,0x3a,0x2e,0x3e,0x1d,
        0x07,0xd6,0x9d,0xe8,0xfc,0xa3,0xe7,0xac,0xc9,0xa0,0xb3,0x22,0x05,0xbc,0xbf,0xd2,
        0x63,0x44,0x66,0xfc,0xb4,0x7b,0xb4,0x70,0x7e,0x96,0xa9,0x16,0x1b,0xb2,0x7d,0x93,
        0x44,0x92,0x5e,0xbd,0x16,0x34,0xa7,0x11,0xd0,0xdf,0x52,0xad,0x6f,0xbd,0x23,0x3c,
        0x3d,0x58,0x16,0xaf,0x99,0x8b,0xbb,0xa0,0xdc,0x3a,0xff,0x17,0xda,0x56,0xba,0x77,
        0xae,0xc4,0xb1,0x51,0xe2,0x61,0x4f,0xf0,0x66,0x1b,0x4c,0xac,0x79,0x34,0x1c,0xfd,
        0x6c,0x5f,0x9a,0x2c,0x60,0xfc,0x47,0x00,0x5f,0x2d,0x81,0xcc,0xa9,0xdd,0x2b,0xf4,
        0x5b,0x53,0x44,0x61,0xd4,0x13,0x5a,0xf3,0x93,0xf0,0xc9,0x24,0xd4,0xe6,0x60,0x6f,
        0x78,0x02,0x0c,0x75,0x9d,0x0d,0x23,0x97,0x35,0xe2,0x06,0x8a,0x49,0x5e,0xe5,0xbe,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4c,0xa7,0x1e,0x02,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x51,0x51,0x64,0x89,0xeb,0x9f,0xf9,0xd6,0xb1,0xa6,0x73,0x5f,0xf1,0x62,0xb5,0xe4,
        0x00,0x80,0xdb,0x4c,0x1c,0xce,0xe5,0x00,0xeb,0xea,0x6c,0x57,0xe4,0x27,0xfc,0x71,
        0x08,0x8c,0xa1,0xfc,0x59,0x1d,0x07,0x45,0x3c,0xc9,0x4e,0x0f,0xb6,0xea,0x96,0x90,
        0xae,0xf7,0x81,0x1e,0x7e,0x6c,0x5e,0x50,0xaf,0x34,0x3e,0xa0,0x55,0x59,0x8e,0xe7,
        0xc1,0xba,0x48,0xfa,0x9e,0x07,0xf6,0x6a,0x24,0x54,0x3e,0x9b,0xa5,0xfe,0x31,0x16,
        0x3d,0x4d,0x9c,0xc4,0xe1,0xec,0x26,0xa0,0x8b,0x59,0xa6,0xf3,0x94,0xf8,0x88,0xda,
        0x1f,0x88,0x23,0x5f,0xfb,0xfd,0x79,0xa2,0xd3,0x62,0x30,0x66,0x69,0xd9,0x0d,0x05,
        0xc0,0x75,0x4c,0xb8,0x48,0x34,0x1d,0x97,0xcf,0x29,0x6a,0x12,0x1c,0x26,0x54,0x1d,
        0x80,0xa9,0x06,0x74,0x86,0xff,0xc6,0xb4,0x72,0xee,0x34,0xe2,0x56,0x06,0x6c,0xf5,
        0x11,0xe7,0x26,0x71,0x47,0x6b,0x05,0xbd,0xe4,0x0b,0x40,0x78,0x84,0x3c,0xf9,0xf2,
        0x78,0x34,0x2b,0x3c,0x5f,0x0e,0x4c,0xfb,0x17,0x39,0xdc,0x59,0x6b,0xd1,0x56,0xac,
        0xe4,0x1f,0xb9,0x19,0xbc,0xec,0xb1,0xd0,0x6d,0x47,0x3b,0x37,0x4d,0x0d,0x6b,0x65,
        0x7c,0x70,0xe9,0xec,0x58,0xcc,0x09,0xd4,0xd9,0xbf,0x9f,0xe0,0x6c,0x7f,0x60,0x28,
        0xd8,0xdf,0x8e,0xd1,0x6a,0x73,0x42,0xf3,0x50,0x01,0x79,0x68,0x41,0xc3,0xba,0x19,
        0x1e,0x2d,0x30,0xc2,0x81,0x2c,0x9f,0x11,0x8b,0xd0,0xdc,0x31,0x3b,0x01,0xfe,0x53,
        0xa5,0x11,0x13,0x22,0x89,0x40,0xb9,0x1b,0x12,0x89,0xef,0x9a,0xcb,0xa8,0x03,0x4f,
        0x54,0x1a,0x15,0x6d,0x11,0xba,0x05,0x09,0xd3,0xdb,0xbf,0x05,0x42,0x3a,0x5a,0x27,
        0x3b,0x34,0x5c,0x58,0x8a,0x5c,0xa4,0xc2,0x28,0xdc,0xb2,0x3a,0xe9,0x99,0x01,0xd6,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x47,0xab,0x9f,0x3a,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x05,0x72,0x44,0xa0,0x99,0x11,0x1d,0x2c,0x4b,0x03,0x4f,0x20,0x92,0x88,0xbe,0x55,
        0xee,0x31,0x2c,0xd9,0xc0,0xc1,0x64,0x77,0x79,0xd7,0x3e,0xfa,0x5a,0x7d,0xf0,0x48,
        0xf8,0xc8,0xfe,0x81,0x8f,0x89,0x92,0xa6,0xc2,0x07,0xdc,0x9f,0x3f,0xb2,0xc8,0xf2,
        0xf3,0xe9,0xe1,0xd3,0xed,0x55,0xb4,0xab,0xc3,0x22,0xed,0x8f,0x00,0xde,0x32,0x95,
        0x91,0xc0,0xc5,0xf3,0xd3,0x93,0xf0,0xee,0x56,0x14,0x8f,0x96,0xff,0xd0,0x6a,0xbd,
        0xfc,0x57,0xc2,0xc3,0x7b,0xc1,0x1d,0x56,0x48,0x3f,0xa6,0xc7,0x92,0x47,0xf7,0x2f,
        0x0b,0x85,0x1c,0xff,0x87,0x29,0xe1,0xbb,0x9b,0x14,0x6c,0xac,0x51,0x0a,0xc0,0x7b,
        0x22,0x25,0xb0,0x48,0x92,0xad,0x09,0x09,0x6e,0x39,0x8e,0x96,0x13,0x05,0x55,0x92,
        0xbd,0xd7,0x5d,0x95,0x35,0xdd,0x8a,0x9d,0x05,0x59,0x60,0xae,0xbb,0xc0,0x85,0x92,
        0x4c,0x8b,0xa0,0x3f,0xa2,0x4a,0xe5,0x2e,0xde,0x85,0x1a,0x39,0x10,0x22,0x11,0x1b,
        0xdd,0xcc,0x96,0xf4,0x93,0x97,0xf5,0x81,0x85,0xf3,0x33,0xda,0xa1,0x9a,0xba,0xfd,
        0xb8,0xaf,0x60,0x81,0x37,0xf1,0x02,0x88,0x54,0x15,0xeb,0x21,0xd1,0x19,0x1a,0x1f,
        0x28,0x9f,0x02,0x27,0xca,0xce,0x97,0xda,0xdc,0xd2,0x0f,0xc5,0x0e,0x2e,0xdd,0x4f,
        0x1d,0x24,0x62,0xe4,0x6e,0x4a,0xbe,0x96,0x95,0x38,0x0c,0xe9,0x26,0x14,0xf3,0xf0,
        0x92,0xbc,0x97,0xdc,0x38,0xeb,0x64,0xc3,0x04,0xc1,0xa2,0x6c,0xad,0xbd,0xf8,0x03,
        0xa0,0xa4,0x68,0xaa,0x9d,0x1f,0x09,0xe6,0x62,0x95,0xa2,0x1c,0x32,0xef,0x62,0x28,
        0x7e,0x54,0x6d,0x4b,0x6a,0xcc,0x4a,0xd0,0x82,0x47,0x46,0x0d,0x45,0x3c,0x36,0x03,
        0x86,0x90,0x44,0x65,0x18,0xac,0x19,0x75,0xe6,0xba,0xb1,0x9a,0xb4,0x5d,0x84,0x9b,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xbc,0xf6,0x46,0x77,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x6e,0x32,0xdb,0x04,0x32,0x57,0x15,0x78,0x0e,0x4c,0x70,0x66,0x5c,0x91,0x43,0x0c,
        0x63,0x73,0xb8,0x86,0xad,0xb0,0xf1,0x34,0x0f,0x0c,0x7e,0xd3,0x4e,0xcb,0xc9,0xea,
        0x19,0x3c,0xb8,0x14,0xd0,0xab,0x9e,0x9b,0x22,0xda,0x7a,0x96,0xa7,0xf5,0xa2,0x99,
        0x58,0xe3,0xd6,0x72,0x0f,0xf5,0xdf,0x88,0xd1,0x33,0xb1,0xe5,0x03,0x72,0x62,0x1c,
        0xa7,0xf2,0x67,0x50,0x0e,0x70,0xc3,0x7a,0x6c,0x4a,0x90,0xba,0x78,0x9e,0xd2,0x0b,
        0x29,0xd4,0xc8,0xa7,0x57,0x06,0xf2,0xf4,0x01,0x4b,0x30,0x53,0xea,0xf7,0xde,0xbf,
        0x1c,0x12,0x03,0xcf,0x9f,0xcf,0x80,0x8b,0x77,0xfd,0x73,0x48,0x79,0x19,0xbe,0x38,
        0x75,0x0b,0x6d,0x78,0x7d,0x79,0x05,0x98,0x65,0x3b,0x35,0x8f,0x68,0xff,0x30,0x7a,
        0x6e,0xf7,0x10,0x9e,0x11,0x25,0xc4,0x95,0x97,0x7d,0x92,0x0f,0xbf,0x38,0x95,0xbd,
        0x5d,0x2a,0xf2,0x06,0x2c,0xd9,0x5a,0x80,0x91,0x4e,0x22,0x7d,0x5f,0x69,0x85,0x03,
        0xa7,0x5d,0xda,0x22,0x09,0x2b,0x8d,0x29,0x67,0x7c,0x8c,0xf6,0xb6,0x49,0x20,0x63,
        0xb9,0xb6,0x4d,0xb6,0x37,0xa3,0x7b,0x19,0xa4,0x28,0x90,0x83,0x55,0x3d,0x4e,0x18,
        0xc8,0x65,0xbc,0xd1,0xe7,0xb5,0xcf,0x65,0x28,0xea,0x19,0x11,0x5c,0xea,0x83,0x8c,
        0x44,0x1f,0xac,0xc5,0xf5,0x3a,0x4b,0x1c,0x2b,0xbf,0x76,0xd8,0x98,0xdb,0x50,0xeb,
        0x64,0x45,0xae,0xa5,0x39,0xb7,0xc8,0xdf,0x5a,0x73,0x6d,0x2d,0x0f,0x4a,0x5a,0x17,
        0x37,0x66,0x1c,0x3d,0x27,0xd5,0xd6,0x7d,0xe1,0x08,0x7f,0xba,0x4d,0x43,0xc2,0x29,
        0xf7,0xbe,0x83,0xec,0xd0,0x3b,0x2e,0x19,0x9e,0xf7,0xbf,0x1b,0x16,0x34,0xd8,0xfa,
        0x32,0x17,0x2a,0x90,0x55,0x93,0xd5,0x3e,0x14,0x8d,0xd6,0xa1,0x40,0x45,0x09,0x52,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x51,0xb2,0x9d,0x4a,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x1d,0xb8,0x77,0xcd,0xcd,0xfe,0xde,0x07,0x97,0xcb,0x97,0x3a,0x4f,0xa0,0xd0,0xe6,
        0xcc,0xcf,0x8b,0x71,0xd5,0x65,0x3d,0xc4,0x17,0x52,0xe7,0x1d,0x6a,0x68,0x4a,0x77,
        0xca,0x04,0x4a,0xef,0x8e,0x7e,0xce,0x79,0xa1,0x80,0x0d,0x9e,0xd5,0xf4,0xce,0x66,
        0x4d,0x54,0xb1,0x09,0xd1,0xb6,0xb0,0x43,0x28,0xe8,0x53,0xe2,0x24,0x9c,0x76,0xc5,
        0x4d,0x22,0xf3,0x6e,0x13,0xf3,0xd7,0xe0,0x85,0xb8,0x9e,0x0b,0x17,0x22,0xc0,0x79,
        0x2b,0x72,0x57,0xaa,0xbd,0x43,0xc3,0xf7,0xde,0xce,0x22,0x41,0x3c,0x7e,0x37,0x1a,
        0x55,0x2e,0x36,0x0e,0x7e,0xdc,0xb3,0xde,0xd7,0x33,0x36,0xc9,0xc8,0x56,0x93,0x51,
        0x68,0x77,0x9a,0xb0,0x08,0x5c,0x22,0x35,0xef,0x5c,0x9b,0xbf,0x3e,0x20,0x8a,0x84,
        0x3d,0xb3,0x60,0x10,0xe1,0x97,0x30,0xd7,0xb3,0x6f,0x40,0x5a,0x2c,0xe0,0xe5,0x52,
        0x19,0xb6,0x2b,0xed,0x6e,0x8e,0x18,0xb4,0x8d,0x78,0xbd,0xc4,0x9f,0x4f,0xbd,0x82,
        0x98,0xd6,0x71,0x3d,0x71,0x5b,0x78,0x73,0xee,0x8e,0x4b,0x37,0x88,0x9e,0x21,0xca,
        0x00,0x6c,0xc2,0x96,0x8d,0xf0,0xcd,0x09,0x58,0x54,0x5a,0x58,0x59,0x8e,0x9b,0xf8,
        0x72,0x93,0xd7,0xa0,0xf9,0xc4,0xdc,0x48,0x89,0xaa,0x31,0x95,0xda,0x4e,0x2f,0x79,
        0x1e,0x37,0x49,0x92,0x2e,0x32,0x2e,0x76,0x54,0x2a,0x64,0xa8,0x96,0x67,0xe9,0x75,
        0x10,0xa6,0xeb,0xad,0xc6,0xa8,0xec,0xb7,0x18,0x0a,0x32,0x26,0x8d,0x6e,0x03,0x74,
        0x0e,0x1f,0xfc,0xde,0x76,0xff,0x6e,0x96,0x42,0x2d,0x80,0x0a,0xc6,0x78,0x70,0xc4,
        0xd8,0x56,0x7b,0xa6,0x38,0x2f,0xf6,0xc0,0x9b,0xd7,0x21,0x6e,0x88,0x5d,0xc8,0xe5,
        0x02,0x6a,0x09,0x1e,0xb3,0x46,0x44,0x80,0x82,0x5b,0xd1,0x66,0x06,0x61,0x4f,0xb8,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xaa,0xef,0x44,0x07,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x42,0x71,0x54,0xe2,0xdb,0x50,0x5d,0x3c,0x10,0xbd,0xf8,0x60,0xbd,0xdb,0x26,0x14,
        0x7d,0x13,0x59,0x98,0x28,0xfb,0x43,0x42,0xca,0x72,0xe6,0xd8,0x58,0x00,0xa2,0x1b,
        0x6a,0x61,0xb4,0x3a,0x80,0x6b,0x9e,0x14,0xbd,0x11,0x33,0xab,0xe9,0xb9,0x91,0x95,
        0xd7,0x5d,0xc3,0x98,0x1f,0x7f,0xcb,0xa8,0xf0,0xec,0x31,0x26,0x51,0xea,0x2e,0xdf,
        0xd9,0xde,0x70,0xf5,0x84,0x27,0x3a,0xac,0x22,0x05,0xb9,0xce,0x2a,0xfb,0x2a,0x83,
        0x1e,0xce,0x0e,0xb2,0x31,0x35,0xc6,0xe6,0xc0,0xd7,0xb0,0x5f,0xf5,0xca,0xdb,0x13,
        0xa7,0xfe,0x4f,0x85,0xa3,0x4f,0x94,0x5c,0xc1,0x04,0x12,0xde,0x6f,0xa1,0xdb,0x41,
        0x59,0x82,0x22,0x22,0x65,0x97,0x6d,0xc8,0x67,0xab,0xf3,0x90,0xeb,0xa4,0x00,0xb3,
        0x7d,0x94,0x3d,0x7b,0x2a,0xe2,0x85,0x36,0x87,0x16,0xb8,0x19,0x92,0x02,0xe0,0x43,
        0x42,0x85,0xa1,0xe6,0xb8,0x11,0x30,0xcc,0x2c,0xd8,0x63,0x09,0x0e,0x53,0x5f,0xa3,
        0xe0,0xd4,0xee,0x0e,0x04,0xee,0x65,0x61,0x96,0x84,0x42,0x0c,0x68,0x8d,0xb7,0x48,
        0xa3,0x02,0xb4,0x82,0x69,0xf2,0x35,0xe4,0xce,0x3b,0xe3,0x44,0xce,0xad,0x49,0x32,
        0xab,0xda,0x04,0xea,0x06,0x60,0xa6,0x2a,0x7d,0xee,0x0f,0xb8,0x95,0x90,0x22,0x62,
        0x9c,0x78,0x59,0xd3,0x7b,0x61,0x02,0x65,0x63,0x96,0x9f,0x67,0x50,0xa0,0x61,0x43,
        0x53,0xb2,0x3f,0x22,0xed,0x8c,0x42,0x39,0x97,0xd9,0xbc,0x6e,0x81,0xb9,0x21,0x97,
        0xc6,0x5b,0x68,0xd7,0x7f,0xd0,0xc5,0x4a,0xfb,0x74,0xc4,0xfd,0x9a,0x2a,0xb8,0x9b,
        0x48,0xe0,0x00,0xea,0x6d,0xf5,0x30,0x26,0x61,0x8f,0xa5,0x45,0x70,0xc9,0x3a,0xea,
        0x6d,0x19,0x11,0x57,0x0f,0x21,0xe6,0x0a,0x53,0x94,0xe3,0x0c,0x99,0xb0,0x2f,0xc5,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xa1,0xe3,0xc5,0x3f,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x41,0xc3,0xcb,0xd7,0x6e,0xde,0x2a,0xc6,0x15,0x05,0xc6,0xba,0x27,0xae,0xcd,0x37,
        0xc0,0xe5,0xbf,0xb9,0x5c,0xdc,0xd6,0xad,0x1a,0xe1,0x35,0x7c,0xc0,0x85,0x85,0x51,
        0x8c,0x98,0x06,0xc0,0x72,0x43,0x71,0x7a,0x2d,0x7c,0x81,0x3c,0xe7,0xd6,0x32,0x8e,
        0x22,0x2b,0x46,0x95,0x6a,0xde,0x45,0x40,0x56,0xe9,0x63,0x32,0x68,0xbf,0xb6,0x78,
        0xb7,0x86,0x00,0x9d,0x2c,0x9e,0xed,0x67,0xc1,0x9b,0x09,0x9e,0xd9,0x0a,0x56,0xcb,
        0x57,0xc9,0x48,0x14,0x23,0x4e,0x97,0x04,0xb5,0x85,0x25,0x1d,0xcb,0x1a,0x79,0x9b,
        0x54,0x06,0x95,0xad,0x16,0x81,0x84,0x3a,0x38,0xec,0x41,0x90,0x2a,0xfa,0x50,0xe0,
        0xb9,0x20,0xa6,0xeb,0xfe,0x2e,0x5c,0xa1,0xf6,0x3c,0x69,0x4c,0xce,0xf8,0x30,0xe0,
        0x87,0x68,0xa2,0x3a,0x9d,0xad,0x75,0xd4,0xa5,0x6b,0x0a,0x90,0x65,0xa2,0x27,0x64,
        0x9d,0xf5,0xa0,0x6f,0xd0,0xd3,0x62,0xa5,0x2d,0xae,0x02,0x89,0xb4,0x1a,0xfa,0x32,
        0x9b,0xa0,0x44,0xdd,0x50,0xde,0xaf,0x41,0xa9,0x89,0x1e,0xb0,0x41,0xbc,0x9c,0x41,
        0xb0,0x35,0x5e,0xf1,0x9a,0xd9,0xab,0x57,0x53,0x21,0xca,0x39,0xfc,0x8b,0xb4,0xd4,
        0xb2,0x19,0x8a,0xe9,0xb2,0x24,0x1e,0xce,0x2e,0x19,0xb0,0xd2,0x93,0x30,0xc4,0x70,
        0xe2,0xf8,0x6a,0x8a,0x99,0x3b,0xed,0x71,0x7e,0x9e,0x98,0x99,0x2a,0xc6,0xdd,0xcf,
        0x43,0x32,0xdb,0xfb,0x27,0x22,0x89,0xa4,0xc5,0xe0,0xa2,0x94,0xe9,0xcf,0x9d,0x48,
        0xab,0x3f,0xfa,0x4f,0x75,0x63,0x46,0xdd,0xfe,0xfa,0xf0,0xbf,0x6e,0xa1,0xf9,0xca,
        0xb1,0x77,0x79,0x35,0x6c,0x33,0xe1,0x57,0x68,0x50,0xe9,0x78,0x4e,0xe4,0xe2,0xf0,
        0xcf,0xe4,0x23,0xde,0xf4,0xa7,0x34,0xb3,0x44,0x97,0x38,0xd2,0xbd,0x27,0x44,0x0e,
      },
      {},
    },
    {
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x5a,0xbe,0x1c,0x72,
      },
      {},
      {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
      },
      {
        0xf1,0x3c,0x7a,0xa4,0xa9,0xaf,0xe7,0x49,0x19,0x7d,0xad,0x50,0xc1,0x6a,0x84,0x87,
        0xf5,0x69,0xe4,0xe5,0xc2,0x0a,0x90,0x33,0xc3,0xeb,0x76,0x63,0x5f,0x9b,0x1d,0xf9,
        0x53,0x4a,0x2a,0x6d,0x6b,0x61,0xe0,0x5d,0xed,0xcb,0x98,0x0d,0xf2,0x57,0x33,0x12,
        0xd1,0x44,0xaa,0x7a,0x7e,0x4e,0x41,0x0e,0xe6,0xa7,0x9f,0x17,0x92,0x28,0x91,0xad,
        0xca,0xce,0xf2,0xa8,0x73,0x4a,0xad,0x89,0x62,0x73,0x0b,0x9a,0x68,0x91,0xa8,0x11,
        0x44,0x01,0xfd,0x57,0xe4,0xf8,0x84,0x55,0x2b,0x66,0xdb,0xb9,0xd6,0xee,0x83,0xe5,
        0x57,0xea,0x5c,0x6a,0x23,0x87,0xdd,0x0a,0x45,0x63,0xb4,0x0c,0x8f,0xc5,0x9f,0x22,
        0xf3,0x4f,0x4e,0x6f,0x7b,0x14,0x62,0xf7,0x80,0x59,0x4a,0xc5,0xc8,0xae,0x8a,0x6f,
        0x5e,0xe3,0x1e,0xe6,0xae,0xec,0x99,0x77,0x6b,0x88,0x14,0xe3,0x58,0x88,0x61,0x74,
        0x38,0x91,0xa1,0x32,0xb8,0xd2,0x39,0x6b,0xe2,0xcb,0x8e,0x77,0xde,0x92,0x36,0x78,
        0xad,0x50,0xcf,0x08,0xb8,0xfa,0x29,0x59,0xb4,0x68,0x1b,0x23,0x10,0x57,0x32,0x92,
        0xf8,0xec,0xe1,0x97,0xdb,0x30,0x85,0x22,0xb5,0x68,0x2f,0xf2,0x98,0xda,0x06,0xee,
        0x65,0x02,0xe7,0xf9,0xc8,0xc1,0xca,0x8f,0xd3,0xed,0x4a,0x3c,0x09,0xdd,0xde,0x64,
        0xd9,0x85,0x17,0x2c,0x62,0x41,0x35,0x24,0xed,0x6b,0x87,0x78,0x1e,0xb5,0x7a,0x9b,
        0xa3,0x90,0xa3,0x99,0xc7,0x39,0x51,0x10,0xb7,0x6a,0x12,0x3b,0x64,0xfe,0x32,0x3c,
        0xb6,0x84,0x9a,0x3f,0x95,0xd3,0xcb,0x22,0x69,0x9c,0xf9,0xb7,0xc2,0x8b,0xf4,0x55,
        0x68,0x60,0x11,0x20,0xc5,0x3e,0x0a,0xc0,0xba,0x00,0x0e,0x88,0x96,0x66,0xfa,0xf0,
        0x75,0xbc,0x2b,0x9c,0xff,0xc5,0x33,0x7b,0xaf,0xb2,0xa6,0x34,0x78,0x44,0x9c,0xa7,
      },
      {},
    },
  };

  for (int i = 0;i < 32;i++)
    {
      auto c = mobius::crypt::new_cipher_gcm ("aes", tv[i].key, tv[i].iv);

      // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      // encryption
      // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      if (tv[i].aad)
        c.authenticate (tv[i].aad);

      auto b = c.encrypt (tv[i].plaintext);
      test.ASSERT_EQUAL (b, tv[i].ciphertext);

      if (tv[i].tag)
        test.ASSERT_TRUE (c.get_tag ().startswith (tv[i].tag));

      // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      // decryption
      // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
      c.reset ();

      if (tv[i].aad)
        c.authenticate (tv[i].aad);

      b = c.decrypt (tv[i].ciphertext);
      test.ASSERT_EQUAL (b, tv[i].plaintext);

      if (tv[i].tag)
        {
          auto siz = tv[i].tag.size ();

          // valid sizes
          if (siz == 16 || siz == 15 || siz == 14 || siz == 13 || siz == 12 || siz == 8 || siz == 4)
            test.ASSERT_TRUE (c.check_tag (tv[i].tag));

          // invalid size. Must fail
          else
            test.ASSERT_FALSE (c.check_tag (tv[i].tag));
        }
    }

  test.end ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \see https://www.schneier.com/code/vectors.txt
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
testcase_cipher_blowfish ()
{
  mobius::unittest test ("mobius::crypt::cipher -> blowfish");

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // mode ECB - key length = 64 bits
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray values1[34][3] =    // key, plaintext, ciphertext
  {
    {
      {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
      {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
      {0x4e, 0xf9, 0x97, 0x45, 0x61, 0x98, 0xdd, 0x78}
    },
    {
      {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
      {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
      {0x51, 0x86, 0x6f, 0xd5, 0xb8, 0x5e, 0xcb, 0x8a}
    },
    {
      {0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
      {0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01},
      {0x7d, 0x85, 0x6f, 0x9a, 0x61, 0x30, 0x63, 0xf2}
    },
    {
      {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
      {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
      {0x24, 0x66, 0xdd, 0x87, 0x8b, 0x96, 0x3c, 0x9d}
    },
    {
      {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
      {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
      {0x61, 0xf9, 0xc3, 0x80, 0x22, 0x81, 0xb0, 0x96}
    },
    {
      {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
      {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
      {0x7d, 0x0c, 0xc6, 0x30, 0xaf, 0xda, 0x1e, 0xc7}
    },
    {
      {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
      {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
      {0x4e, 0xf9, 0x97, 0x45, 0x61, 0x98, 0xdd, 0x78}
    },
    {
      {0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},
      {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
      {0x0a, 0xce, 0xab, 0x0f, 0xc6, 0xa0, 0xa2, 0x8d}
    },
    {
      {0x7c, 0xa1, 0x10, 0x45, 0x4a, 0x1a, 0x6e, 0x57},
      {0x01, 0xa1, 0xd6, 0xd0, 0x39, 0x77, 0x67, 0x42},
      {0x59, 0xc6, 0x82, 0x45, 0xeb, 0x05, 0x28, 0x2b}
    },
    {
      {0x01, 0x31, 0xd9, 0x61, 0x9d, 0xc1, 0x37, 0x6e},
      {0x5c, 0xd5, 0x4c, 0xa8, 0x3d, 0xef, 0x57, 0xda},
      {0xb1, 0xb8, 0xcc, 0x0b, 0x25, 0x0f, 0x09, 0xa0}
    },
    {
      {0x07, 0xa1, 0x13, 0x3e, 0x4a, 0x0b, 0x26, 0x86},
      {0x02, 0x48, 0xd4, 0x38, 0x06, 0xf6, 0x71, 0x72},
      {0x17, 0x30, 0xe5, 0x77, 0x8b, 0xea, 0x1d, 0xa4}
    },
    {
      {0x38, 0x49, 0x67, 0x4c, 0x26, 0x02, 0x31, 0x9e},
      {0x51, 0x45, 0x4b, 0x58, 0x2d, 0xdf, 0x44, 0x0a},
      {0xa2, 0x5e, 0x78, 0x56, 0xcf, 0x26, 0x51, 0xeb}
    },
    {
      {0x04, 0xb9, 0x15, 0xba, 0x43, 0xfe, 0xb5, 0xb6},
      {0x42, 0xfd, 0x44, 0x30, 0x59, 0x57, 0x7f, 0xa2},
      {0x35, 0x38, 0x82, 0xb1, 0x09, 0xce, 0x8f, 0x1a}
    },
    {
      {0x01, 0x13, 0xb9, 0x70, 0xfd, 0x34, 0xf2, 0xce},
      {0x05, 0x9b, 0x5e, 0x08, 0x51, 0xcf, 0x14, 0x3a},
      {0x48, 0xf4, 0xd0, 0x88, 0x4c, 0x37, 0x99, 0x18}
    },
    {
      {0x01, 0x70, 0xf1, 0x75, 0x46, 0x8f, 0xb5, 0xe6},
      {0x07, 0x56, 0xd8, 0xe0, 0x77, 0x47, 0x61, 0xd2},
      {0x43, 0x21, 0x93, 0xb7, 0x89, 0x51, 0xfc, 0x98}
    },
    {
      {0x43, 0x29, 0x7f, 0xad, 0x38, 0xe3, 0x73, 0xfe},
      {0x76, 0x25, 0x14, 0xb8, 0x29, 0xbf, 0x48, 0x6a},
      {0x13, 0xf0, 0x41, 0x54, 0xd6, 0x9d, 0x1a, 0xe5}
    },
    {
      {0x07, 0xa7, 0x13, 0x70, 0x45, 0xda, 0x2a, 0x16},
      {0x3b, 0xdd, 0x11, 0x90, 0x49, 0x37, 0x28, 0x02},
      {0x2e, 0xed, 0xda, 0x93, 0xff, 0xd3, 0x9c, 0x79}
    },
    {
      {0x04, 0x68, 0x91, 0x04, 0xc2, 0xfd, 0x3b, 0x2f},
      {0x26, 0x95, 0x5f, 0x68, 0x35, 0xaf, 0x60, 0x9a},
      {0xd8, 0x87, 0xe0, 0x39, 0x3c, 0x2d, 0xa6, 0xe3}
    },
    {
      {0x37, 0xd0, 0x6b, 0xb5, 0x16, 0xcb, 0x75, 0x46},
      {0x16, 0x4d, 0x5e, 0x40, 0x4f, 0x27, 0x52, 0x32},
      {0x5f, 0x99, 0xd0, 0x4f, 0x5b, 0x16, 0x39, 0x69}
    },
    {
      {0x1f, 0x08, 0x26, 0x0d, 0x1a, 0xc2, 0x46, 0x5e},
      {0x6b, 0x05, 0x6e, 0x18, 0x75, 0x9f, 0x5c, 0xca},
      {0x4a, 0x05, 0x7a, 0x3b, 0x24, 0xd3, 0x97, 0x7b}
    },
    {
      {0x58, 0x40, 0x23, 0x64, 0x1a, 0xba, 0x61, 0x76},
      {0x00, 0x4b, 0xd6, 0xef, 0x09, 0x17, 0x60, 0x62},
      {0x45, 0x20, 0x31, 0xc1, 0xe4, 0xfa, 0xda, 0x8e}
    },
    {
      {0x02, 0x58, 0x16, 0x16, 0x46, 0x29, 0xb0, 0x07},
      {0x48, 0x0d, 0x39, 0x00, 0x6e, 0xe7, 0x62, 0xf2},
      {0x75, 0x55, 0xae, 0x39, 0xf5, 0x9b, 0x87, 0xbd}
    },
    {
      {0x49, 0x79, 0x3e, 0xbc, 0x79, 0xb3, 0x25, 0x8f},
      {0x43, 0x75, 0x40, 0xc8, 0x69, 0x8f, 0x3c, 0xfa},
      {0x53, 0xc5, 0x5f, 0x9c, 0xb4, 0x9f, 0xc0, 0x19}
    },
    {
      {0x4f, 0xb0, 0x5e, 0x15, 0x15, 0xab, 0x73, 0xa7},
      {0x07, 0x2d, 0x43, 0xa0, 0x77, 0x07, 0x52, 0x92},
      {0x7a, 0x8e, 0x7b, 0xfa, 0x93, 0x7e, 0x89, 0xa3}
    },
    {
      {0x49, 0xe9, 0x5d, 0x6d, 0x4c, 0xa2, 0x29, 0xbf},
      {0x02, 0xfe, 0x55, 0x77, 0x81, 0x17, 0xf1, 0x2a},
      {0xcf, 0x9c, 0x5d, 0x7a, 0x49, 0x86, 0xad, 0xb5}
    },
    {
      {0x01, 0x83, 0x10, 0xdc, 0x40, 0x9b, 0x26, 0xd6},
      {0x1d, 0x9d, 0x5c, 0x50, 0x18, 0xf7, 0x28, 0xc2},
      {0xd1, 0xab, 0xb2, 0x90, 0x65, 0x8b, 0xc7, 0x78}
    },
    {
      {0x1c, 0x58, 0x7f, 0x1c, 0x13, 0x92, 0x4f, 0xef},
      {0x30, 0x55, 0x32, 0x28, 0x6d, 0x6f, 0x29, 0x5a},
      {0x55, 0xcb, 0x37, 0x74, 0xd1, 0x3e, 0xf2, 0x01}
    },
    {
      {0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},
      {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
      {0xfa, 0x34, 0xec, 0x48, 0x47, 0xb2, 0x68, 0xb2}
    },
    {
      {0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e},
      {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
      {0xa7, 0x90, 0x79, 0x51, 0x08, 0xea, 0x3c, 0xae}
    },
    {
      {0xe0, 0xfe, 0xe0, 0xfe, 0xf1, 0xfe, 0xf1, 0xfe},
      {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
      {0xc3, 0x9e, 0x07, 0x2d, 0x9f, 0xac, 0x63, 0x1d}
    },
    {
      {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
      {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
      {0x01, 0x49, 0x33, 0xe0, 0xcd, 0xaf, 0xf6, 0xe4}
    },
    {
      {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
      {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
      {0xf2, 0x1e, 0x9a, 0x77, 0xb7, 0x1c, 0x49, 0xbc}
    },
    {
      {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
      {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
      {0x24, 0x59, 0x46, 0x88, 0x57, 0x54, 0x36, 0x9a}
    },
    {
      {0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},
      {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
      {0x6b, 0x5c, 0x5a, 0x9c, 0x5d, 0x9e, 0x0a, 0x5a}
    },
  };

  for (int i = 0; i < 34; i++)
    {
      auto cipher = mobius::crypt::new_cipher_ecb ("blowfish", values1[i][0]);
      test.ASSERT_EQUAL (cipher.encrypt (values1[i][1]), values1[i][2]);
      test.ASSERT_EQUAL (cipher.decrypt (values1[i][2]), values1[i][1]);
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // mode ECB - variable key length
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  const mobius::bytearray plaintext = { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 };

  const mobius::bytearray values2[24][2] =    // key, ciphertext
  {
    {
      {0xf0},
      {0xf9, 0xad, 0x59, 0x7c, 0x49, 0xdb, 0x00, 0x5e}
    },
    {
      {0xf0, 0xe1},
      {0xe9, 0x1d, 0x21, 0xc1, 0xd9, 0x61, 0xa6, 0xd6}
    },
    {
      {0xf0, 0xe1, 0xd2},
      {0xe9, 0xc2, 0xb7, 0x0a, 0x1b, 0xc6, 0x5c, 0xf3}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3},
      {0xbe, 0x1e, 0x63, 0x94, 0x08, 0x64, 0x0f, 0x05}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4},
      {0xb3, 0x9e, 0x44, 0x48, 0x1b, 0xdb, 0x1e, 0x6e}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5},
      {0x94, 0x57, 0xaa, 0x83, 0xb1, 0x92, 0x8c, 0x0d}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96},
      {0x8b, 0xb7, 0x70, 0x32, 0xf9, 0x60, 0x62, 0x9d}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87},
      {0xe8, 0x7a, 0x24, 0x4e, 0x2c, 0xc8, 0x5e, 0x82}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78},
      {0x15, 0x75, 0x0e, 0x7a, 0x4f, 0x4e, 0xc5, 0x77}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69},
      {0x12, 0x2b, 0xa7, 0x0b, 0x3a, 0xb6, 0x4a, 0xe0}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a},
      {0x3a, 0x83, 0x3c, 0x9a, 0xff, 0xc5, 0x37, 0xf6}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b},
      {0x94, 0x09, 0xda, 0x87, 0xa9, 0x0f, 0x6b, 0xf2}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c},
      {0x88, 0x4f, 0x80, 0x62, 0x50, 0x60, 0xb8, 0xb4}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d},
      {0x1f, 0x85, 0x03, 0x1c, 0x19, 0xe1, 0x19, 0x68}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e},
      {0x79, 0xd9, 0x37, 0x3a, 0x71, 0x4c, 0xa3, 0x4f}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f},
      {0x93, 0x14, 0x28, 0x87, 0xee, 0x3b, 0xe1, 0x5c}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00},
      {0x03, 0x42, 0x9e, 0x83, 0x8c, 0xe2, 0xd1, 0x4b}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00, 0x11},
      {0xa4, 0x29, 0x9e, 0x27, 0x46, 0x9f, 0xf6, 0x7b}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00, 0x11, 0x22},
      {0xaf, 0xd5, 0xae, 0xd1, 0xc1, 0xbc, 0x96, 0xa8}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00, 0x11, 0x22, 0x33},
      {0x10, 0x85, 0x1c, 0x0e, 0x38, 0x58, 0xda, 0x9f}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00, 0x11, 0x22, 0x33, 0x44},
      {0xe6, 0xf5, 0x1e, 0xd7, 0x9b, 0x9d, 0xb2, 0x1f}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55},
      {0x64, 0xa6, 0xe1, 0x4a, 0xfd, 0x36, 0xb4, 0x6f}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66},
      {0x80, 0xc7, 0xd7, 0xd4, 0x5a, 0x54, 0x79, 0xad}
    },
    {
      {0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87, 0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77},
      {0x05, 0x04, 0x4b, 0x62, 0xfa, 0x52, 0xd0, 0x80}
    },
  };

  for (int i = 0; i < 24; i++)
    {
      auto cipher = mobius::crypt::new_cipher_ecb ("blowfish", values2[i][0]);
      test.ASSERT_EQUAL (cipher.encrypt (plaintext), values2[i][1]);
      test.ASSERT_EQUAL (cipher.decrypt (values2[i][1]), plaintext);
    }

   // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // https://www.schneier.com/wp-content/uploads/2015/12/vectors-2.txt
  // Blowfish-CBC
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto cm = mobius::crypt::new_cipher_cbc ("blowfish",
         {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef, 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87},
         {0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10}
  );

  const mobius::bytearray blowfish_plaintext =
  {
    0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20, 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x00, 0x00, 0x00, 0x00
  };

  const mobius::bytearray blowfish_ciphertext =
  {
    0x6b, 0x77, 0xb4, 0xd6, 0x30, 0x06, 0xde, 0xe6, 0x05, 0xb1, 0x56, 0xe2, 0x74, 0x03, 0x97, 0x93,
    0x58, 0xde, 0xb9, 0xe7, 0x15, 0x46, 0x16, 0xd9, 0x59, 0xf1, 0x65, 0x2b, 0xd5, 0xff, 0x92, 0xcc
  };

  auto b = cm.encrypt (blowfish_plaintext);
  test.ASSERT_EQUAL (b, blowfish_ciphertext);

  cm.reset ();
  b = cm.decrypt (blowfish_ciphertext);
  test.ASSERT_EQUAL (b, blowfish_plaintext);

  test.end ();
}

static void
testcase_cipher_des ()
{
  // DES - mode ECB
  auto des0 = mobius::crypt::new_cipher_ecb ("des", {0x00, 0xe8, 0x03, 0x00, 0x00, 0xe8, 0x03});
  mobius::bytearray b0 = {0x33, 0xb8, 0x17, 0xb0, 0x9b, 0x82, 0xa4, 0xf1};

  mobius::unittest test ("mobius::crypt::cipher -> des");
  test.ASSERT_EQUAL (b0.to_hexstring (), "33b817b09b82a4f1");

  b0 = des0.decrypt (b0);
  test.ASSERT_EQUAL (b0.to_hexstring (), "f4ed3a422ae3a1a6");

  // DES - mode ECB
  auto des1 = mobius::crypt::new_cipher_ecb ("des", {0x13, 0x34, 0x57, 0x79, 0x9b, 0xbc, 0xdf, 0xf1});
  mobius::bytearray b1 = {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef};

  test.ASSERT_EQUAL (b1.to_hexstring (), "0123456789abcdef");

  b1 = des1.encrypt (b1);
  test.ASSERT_EQUAL (b1.to_hexstring (), "85e813540f0ab405");

  b1 = des1.decrypt (b1);
  test.ASSERT_EQUAL (b1.to_hexstring (), "0123456789abcdef");

  // DES - mode ECB
  auto des2 = mobius::crypt::new_cipher_ecb ("des", {0x0e, 0x32, 0x92, 0x32, 0xea, 0x6d, 0x0d, 0x73});
  mobius::bytearray b2 (8);
  b2.fill (0x87);

  test.ASSERT_EQUAL (b2.to_hexstring (), "8787878787878787");

  b2 = des2.encrypt (b2);
  test.ASSERT_EQUAL (b2.to_hexstring (), "0000000000000000");

  b2 = des2.decrypt (b2);
  test.ASSERT_EQUAL (b2.to_hexstring (), "8787878787878787");

  // LM password "ABCD123"
  auto des3 = mobius::crypt::new_cipher_ecb ("des", "ABCD123");
  mobius::bytearray b3 ("KGS!@#$%");

  test.ASSERT_EQUAL (b3.to_hexstring (), "4b47532140232425");

  b3 = des3.encrypt (b3);
  test.ASSERT_EQUAL (b3.to_hexstring (), "6f87cd328120cc55");

  b3 = des3.decrypt (b3);
  test.ASSERT_EQUAL (b3.to_hexstring (), "4b47532140232425");

  // LM password "ABCDE12"
  auto des4 = mobius::crypt::new_cipher_ecb ("des", "ABCDE12");
  mobius::bytearray b4 ("KGS!@#$%");

  test.ASSERT_EQUAL (b4.to_hexstring (), "4b47532140232425");

  b4 = des4.encrypt (b4);
  test.ASSERT_EQUAL (b4.to_hexstring (), "722ac01404a75156");

  b4 = des4.decrypt (b4);
  test.ASSERT_EQUAL (b4.to_hexstring (), "4b47532140232425");

  // DES - mode ECB
  auto des6 = mobius::crypt::new_cipher_ecb ("des", {0xe7, 0x37, 0x4e, 0x45, 0x8d, 0x54, 0xfd});
  mobius::bytearray b6 = {0x75, 0x3f, 0x61, 0x59, 0xc3, 0xa3, 0x11, 0x49};

  test.ASSERT_EQUAL (b6.to_hexstring (), "753f6159c3a31149");

  b6 = des6.decrypt (b6);
  test.ASSERT_EQUAL (b6.to_hexstring (), "0000000001000000");

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // DES - mode CBC
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto cm = mobius::crypt::new_cipher_cbc ("des", "ABCDE12");

  mobius::bytearray des_plaintext_1 ("KGS!@#$%");
  mobius::bytearray des_ciphertext_1 = {0x72, 0x2a, 0xc0, 0x14, 0x04, 0xa7, 0x51, 0x56};

  auto b = cm.encrypt (des_plaintext_1);
  test.ASSERT_EQUAL (b, des_ciphertext_1);

  cm.reset ();
  b = cm.decrypt (des_ciphertext_1);
  test.ASSERT_EQUAL (b, des_plaintext_1);

  cm = mobius::crypt::new_cipher_cbc ("des", {0x17,0xc0,0xfc,0x69,0x31,0x10,0xcb,0xd7});

  const mobius::bytearray des_plaintext_2 =
  {
    0xe5, 0xa3, 0xac, 0x4e, 0x83, 0xfa, 0x58, 0x62,
    0xf5, 0x9f, 0x57, 0x73, 0x98, 0x1c, 0x70, 0x03,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
  };

  const mobius::bytearray des_ciphertext_2 =
  {
    0x62, 0x75, 0x32, 0xeb, 0xd7, 0x97, 0xe0, 0xfe,
    0x7d, 0xf4, 0x7c, 0x02, 0xa5, 0x12, 0x61, 0x3e,
    0x9f, 0xdb, 0xd3, 0x7e, 0x8b, 0x89, 0x2d, 0xc1
  };

  b = cm.encrypt (des_plaintext_2);
  test.ASSERT_EQUAL (b, des_ciphertext_2);

  cm.reset ();
  b = cm.decrypt (des_ciphertext_2);
  test.ASSERT_EQUAL (b, des_plaintext_2);

  test.end ();
}

static void
testcase_cipher_rc2 ()
{
  mobius::unittest test ("mobius::crypt::cipher -> rc2");

  // RFC 2268, section 5
  auto rc2_1 = mobius::crypt::new_cipher_ecb ("rc2", {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff});
  mobius::bytearray b_1 = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

  test.ASSERT_EQUAL (b_1.to_hexstring (), "ffffffffffffffff");
  b_1 = rc2_1.encrypt (b_1);
  test.ASSERT_EQUAL (b_1.to_hexstring (), "278b27e42e2f0d49");
  b_1 = rc2_1.decrypt (b_1);
  test.ASSERT_EQUAL (b_1.to_hexstring (), "ffffffffffffffff");

  // RFC 2268, section 5
  auto rc2_2 = mobius::crypt::new_cipher_ecb ("rc2", {0x30, 0, 0, 0, 0, 0, 0, 0});
  mobius::bytearray b_2 = {0x10, 0, 0, 0, 0, 0, 0, 0x01};

  test.ASSERT_EQUAL (b_2.to_hexstring (), "1000000000000001");
  b_2 = rc2_2.encrypt (b_2);
  test.ASSERT_EQUAL (b_2.to_hexstring (), "30649edf9be7d2c2");
  b_2 = rc2_2.decrypt (b_2);
  test.ASSERT_EQUAL (b_2.to_hexstring (), "1000000000000001");

  // RFC 2268, section 5
  auto rc2_3 = mobius::crypt::new_cipher_ecb ("rc2", {0x88, 0xbc, 0xa9, 0x0e, 0x90, 0x87, 0x5a, 0x7f, 0x0f, 0x79, 0xc3, 0x84, 0x62, 0x7b, 0xaf, 0xb2});
  mobius::bytearray b_3 = {0, 0, 0, 0, 0, 0, 0, 0};

  test.ASSERT_EQUAL (b_3.to_hexstring (), "0000000000000000");
  b_3 = rc2_3.encrypt (b_3);
  test.ASSERT_EQUAL (b_3.to_hexstring (), "2269552ab0f85ca6");
  b_3 = rc2_3.decrypt (b_3);
  test.ASSERT_EQUAL (b_3.to_hexstring (), "0000000000000000");

  test.end ();
}

static void
testcase_cipher_rc4 ()
{
  mobius::unittest test ("mobius::crypt::cipher -> rc4");

  // Test vector from Cryptlib
  auto rc4_1 = mobius::crypt::new_cipher_stream ("rc4", {0x61, 0x8A, 0x63, 0xD2, 0xFB});
  const mobius::bytearray plaintext_1 = {0xDC, 0xEE, 0x4C, 0xF9, 0x2C};
  const mobius::bytearray ciphertext_1 = {0xF1, 0x38, 0x29, 0xC9, 0xDE};

  auto b = rc4_1.encrypt (plaintext_1);
  test.ASSERT_EQUAL (b, ciphertext_1);
  rc4_1.reset ();
  b = rc4_1.decrypt (ciphertext_1);
  test.ASSERT_EQUAL (b, plaintext_1);

  // key="Secret", Plaintext="Attack at dawn"
  auto rc4_2 = mobius::crypt::new_cipher_stream ("rc4", "Secret");
  const mobius::bytearray plaintext_2 = "Attack at dawn";
  const mobius::bytearray ciphertext_2 = {0x45, 0xa0, 0x1f, 0x64, 0x5f, 0xc3, 0x5b,
  0x38, 0x35, 0x52, 0x54, 0x4b, 0x9b, 0xf5};

  b = rc4_2.encrypt (plaintext_2);
  test.ASSERT_EQUAL (b, ciphertext_2);
  rc4_2.reset ();
  b = rc4_2.decrypt (ciphertext_2);
  test.ASSERT_EQUAL (b, plaintext_2);
  test.end ();
}

static void
testcase_cipher_zip ()
{
  mobius::unittest test ("mobius::crypt::cipher -> zip");

  // key="Key", Plaintext="Plaintext"
  auto zip_1 = mobius::crypt::new_cipher_stream ("zip", "Key");
  mobius::bytearray b_1 = "Plaintext";

  test.ASSERT_EQUAL (b_1.to_hexstring (), "506c61696e74657874");
  b_1 = zip_1.encrypt (b_1);
  test.ASSERT_EQUAL (b_1.to_hexstring (), "fe1995e4fe54a8c6f3");
  zip_1.reset ();
  b_1 = zip_1.decrypt (b_1);
  test.ASSERT_EQUAL (b_1.to_hexstring (), "506c61696e74657874");

  // key="Secret", Plaintext="Attack at dawn"
  auto zip_2 = mobius::crypt::new_cipher_stream ("zip", "Secret");
  mobius::bytearray b_2 = "Attack at dawn";

  test.ASSERT_EQUAL (b_2.to_hexstring (), "41747461636b206174206461776e");
  b_2 = zip_2.encrypt (b_2);
  test.ASSERT_EQUAL (b_2.to_hexstring (), "7595da02f5ec5c2c78755fd4069e");
  zip_2.reset ();
  b_2 = zip_2.decrypt (b_2);
  test.ASSERT_EQUAL (b_2.to_hexstring (), "41747461636b206174206461776e");
  test.end ();
}

static void
testcase_hash_adler32 ()
{
  mobius::unittest test ("mobius::crypt::hash -> adler32");

  mobius::crypt::hash h1 ("adler32");
  mobius::crypt::hash h2 ("adler32");
  mobius::crypt::hash h3 ("adler32");

  test.ASSERT_EQUAL (h1.get_hex_digest (), "00000001");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "00000001");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "00000001");

  // update
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "024d0127");

  h2.update ("d");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "03d8018b");

  h3.update ("abcd");
  test.ASSERT_EQUAL (h3.get_hex_digest (), h2.get_hex_digest ());

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("adler32");
  h100.update ("a");
  h100.get_digest ();
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("adler32");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_crc32 ()
{
  mobius::unittest test ("mobius::crypt::hash -> crc32");

  mobius::crypt::hash h1 ("crc32");
  mobius::crypt::hash h2 ("crc32");
  mobius::crypt::hash h3 ("crc32");

  test.ASSERT_EQUAL (h1.get_hex_digest (), "00000000");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "00000000");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "00000000");

  // update
  h2.reset ();
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "352441c2");

  h2.reset ();
  h2.update ("abc");
  h2.update ("d");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "ed82cd11");

  h3.reset ();
  h3.update ("abcd");
  test.ASSERT_EQUAL (h3.get_hex_digest (), h2.get_hex_digest ());

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("crc32");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("crc32");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_ed2k ()
{
  mobius::unittest test ("mobius::crypt::hash -> ed2k");

  // Same as hash_md4 for data.size <= 9.728.000
  mobius::crypt::hash h1 ("ed2k");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "31d6cfe0d16ae931b73c59d7e0c089c0");

  mobius::crypt::hash h2 ("ed2k");
  h2.update ("a");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "bde52cb31de33e46245e05fbdbd6fb24");

  mobius::crypt::hash h3 ("ed2k");
  h3.update ("abc");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "a448017aaf21d8525fc10ae87aa6729d");

  mobius::crypt::hash h4 ("ed2k");
  h4.update ("message digest");
  test.ASSERT_EQUAL (h4.get_hex_digest (), "d9130a8164549fe818874806e1c7014b");

  mobius::crypt::hash h5 ("ed2k");
  h5.update ("abcdefghijklmnopqrstuvwxyz");
  test.ASSERT_EQUAL (h5.get_hex_digest (), "d79e1c308aa5bbcdeea8ed63df412da9");

  mobius::crypt::hash h6 ("ed2k");
  h6.update ("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
  test.ASSERT_EQUAL (h6.get_hex_digest (), "043f8582f241db351ce627e153e7f0e4");

  mobius::crypt::hash h7 ("ed2k");
  h7.update ("12345678901234567890123456789012345678901234567890123456789012345678901234567890");
  test.ASSERT_EQUAL (h7.get_hex_digest (), "e33b4ddc9c38f2199c3e7b164fcc0536");

  // data.size = 9.728.000
  mobius::crypt::hash h8 ("ed2k");
  mobius::bytearray d8 (9728000);
  d8.fill (0);
  h8.update (d8);
  test.ASSERT_EQUAL (h8.get_hex_digest (), "d7def262a127cd79096a108e7a9fc138");

  // data.size = 9.728.000 - 1
  mobius::crypt::hash h9 ("ed2k");
  mobius::bytearray d9 (9728000 - 1);
  d9.fill (0);
  h9.update (d9);
  test.ASSERT_EQUAL (h9.get_hex_digest (), "ac44b93fc9aff773ab0005c911f8396f");

  // data.size = 9.728.000 + 1
  mobius::crypt::hash h10 ("ed2k");
  mobius::bytearray d10 (9728000 + 1);
  d10.fill (0);
  h10.update (d10);
  test.ASSERT_EQUAL (h10.get_hex_digest (), "06329e9dba1373512c06386fe29e3c65");

  // data.size = 9.728.000 - 1 + 0 + 0
  mobius::crypt::hash h11 ("ed2k");
  mobius::bytearray d11 (9728000 - 1);
  d11.fill (0);
  h11.update (d11);
  h11.update ({0});
  h11.update ({0});
  test.ASSERT_EQUAL (h11.get_hex_digest (), "06329e9dba1373512c06386fe29e3c65");

  // data.size = 9.728.000 - 1 + 0 + 0
  mobius::crypt::hash h12 ("ed2k");
  h12.update ({0});
  h12.update ({0});
  h12.update (d11);
  test.ASSERT_EQUAL (h12.get_hex_digest (), "06329e9dba1373512c06386fe29e3c65");

  // data.size = 9.728.001
  mobius::crypt::hash h13 ("ed2k");
  mobius::bytearray d13 = {0};

  for (auto i = 0;i < 9728001;i++)
    h13.update (d13);

  test.ASSERT_EQUAL (h13.get_hex_digest (), "06329e9dba1373512c06386fe29e3c65");

  // update ("a"), update ("b") = update ("ab")
  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("ed2k");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("ed2k");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (d100, d101);

  test.end ();
}

static void
testcase_hash_md4 ()
{
  mobius::unittest test ("mobius::crypt::hash -> md4");

  // RFC 1320 - section A.5
  mobius::crypt::hash h1 ("md4");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "31d6cfe0d16ae931b73c59d7e0c089c0");

  mobius::crypt::hash h2 ("md4");
  h2.update ("a");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "bde52cb31de33e46245e05fbdbd6fb24");

  mobius::crypt::hash h3 ("md4");
  h3.update ("abc");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "a448017aaf21d8525fc10ae87aa6729d");

  mobius::crypt::hash h4 ("md4");
  h4.update ("message digest");
  test.ASSERT_EQUAL (h4.get_hex_digest (), "d9130a8164549fe818874806e1c7014b");

  mobius::crypt::hash h5 ("md4");
  h5.update ("abcdefghijklmnopqrstuvwxyz");
  test.ASSERT_EQUAL (h5.get_hex_digest (), "d79e1c308aa5bbcdeea8ed63df412da9");

  mobius::crypt::hash h6 ("md4");
  h6.update ("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
  test.ASSERT_EQUAL (h6.get_hex_digest (), "043f8582f241db351ce627e153e7f0e4");

  mobius::crypt::hash h7 ("md4");
  h7.update ("12345678901234567890123456789012345678901234567890123456789012345678901234567890");
  test.ASSERT_EQUAL (h7.get_hex_digest (), "e33b4ddc9c38f2199c3e7b164fcc0536");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("md4");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("md4");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (d100, d101);

  test.end ();
}

static void
testcase_hash_md5 ()
{
  mobius::unittest test ("mobius::crypt::hash -> md5");

  // RFC 1321 - section A.5
  mobius::crypt::hash h1 ("md5");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "d41d8cd98f00b204e9800998ecf8427e");

  mobius::crypt::hash h2 ("md5");
  h2.update ("a");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "0cc175b9c0f1b6a831c399e269772661");

  mobius::crypt::hash h3 ("md5");
  h3.update ("abc");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "900150983cd24fb0d6963f7d28e17f72");

  mobius::crypt::hash h4 ("md5");
  h4.update ("message digest");
  test.ASSERT_EQUAL (h4.get_hex_digest (), "f96b697d7cb7938d525a2f31aaf161d0");

  mobius::crypt::hash h5 ("md5");
  h5.update ("abcdefghijklmnopqrstuvwxyz");
  test.ASSERT_EQUAL (h5.get_hex_digest (), "c3fcd3d76192e4007dfb496cca67e13b");

  mobius::crypt::hash h6 ("md5");
  h6.update ("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
  test.ASSERT_EQUAL (h6.get_hex_digest (), "d174ab98d277d9f5a5611c2c9f419d9f");

  mobius::crypt::hash h7 ("md5");
  h7.update ("12345678901234567890123456789012345678901234567890123456789012345678901234567890");
  test.ASSERT_EQUAL (h7.get_hex_digest (), "57edf4a22be3c955ac49da2e2107b67a");

  mobius::crypt::hash h8 ("md5");
  mobius::bytearray b1 (1000);
  b1.fill (0);
  h8.update (b1);
  test.ASSERT_EQUAL (h8.get_hex_digest (), "ede3d3b685b4e137ba4cb2521329a75e");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("md5");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("md5");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (d100, d101);

  test.end ();
}

static void
testcase_hash_sha1 ()
{
  mobius::unittest test ("mobius::crypt::hash -> sha1");

  // RFC 3174 - section 7.3
  mobius::crypt::hash h1 ("sha1");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "da39a3ee5e6b4b0d3255bfef95601890afd80709");

  mobius::crypt::hash h2 ("sha1");
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "a9993e364706816aba3e25717850c26c9cd0d89d");

  mobius::crypt::hash h3 ("sha1");
  h3.update ("abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "84983e441c3bd26ebaae4aa1f95129e5e54670f1");

  mobius::crypt::hash h4 ("sha1");

  for (int i = 0; i < 1000000; i++)
    h4.update ("a");

  test.ASSERT_EQUAL (h4.get_hex_digest (), "34aa973cd4c4daa4f61eeb2bdbad27316534016f");

  mobius::crypt::hash h5 ("sha1");

  for (int i = 0; i < 20; i++)
    h5.update ("01234567012345670123456701234567");

  test.ASSERT_EQUAL (h5.get_hex_digest (), "dea356a2cddd90c7a7ecedc5ebb563934f460452");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("sha1");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("sha1");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_sha2_224 ()
{
  mobius::unittest test ("mobius::crypt::hash -> sha2-224");

  mobius::crypt::hash h1 ("sha2-224");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "d14a028c2a3a2bc9476102bb288234c415a2b01f828ea62ac5b3e42f");

  //! \see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA224.pdf
  mobius::crypt::hash h2 ("sha2-224");
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "23097d223405d8228642a477bda255b32aadbce4bda0b3f7e36c9da7");

  mobius::crypt::hash h3 ("sha2-224");
  h3.update ("abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "75388b16512776cc5dba5da1fd890150b0c6455cb4f58b1952522525");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("sha2-224");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("sha2-224");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_sha2_256 ()
{
  mobius::unittest test ("mobius::crypt::hash -> sha2-256");

  mobius::crypt::hash h1 ("sha2-256");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855");

  //! \see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA256.pdf
  mobius::crypt::hash h2 ("sha2-256");
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "ba7816bf8f01cfea414140de5dae2223b00361a396177a9cb410ff61f20015ad");

  mobius::crypt::hash h3 ("sha2-256");
  h3.update ("abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "248d6a61d20638b8e5c026930c3e6039a33ce45964ff2167f6ecedd419db06c1");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("sha2-256");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("sha2-256");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_sha2_384 ()
{
  mobius::unittest test ("mobius::crypt::hash -> sha2-384");

  mobius::crypt::hash h1 ("sha2-384");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "38b060a751ac96384cd9327eb1b1e36a21fdb71114be07434c0cc7bf63f6e1da274edebfe76f65fbd51ad2f14898b95b");

  //! \see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA384.pdf
  mobius::crypt::hash h2 ("sha2-384");
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "cb00753f45a35e8bb5a03d699ac65007272c32ab0eded1631a8b605a43ff5bed8086072ba1e7cc2358baeca134c825a7");

  mobius::crypt::hash h3 ("sha2-384");
  h3.update ( "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "09330c33f71147e83d192fc782cd1b4753111b173b3b05d22fa08086e3b0f712fcc7c71a557e2db966c3e9fa91746039");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("sha2-384");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("sha2-384");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_sha2_512 ()
{
  mobius::unittest test ("mobius::crypt::hash -> sha2-512");

  mobius::crypt::hash h1 ("sha2-512");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "cf83e1357eefb8bdf1542850d66d8007d620e4050b5715dc83f4a921d36ce9ce47d0d13c5d85f2b0ff8318d2877eec2f63b931bd47417a81a538327af927da3e");

  //! \see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA512.pdf
  mobius::crypt::hash h2 ("sha2-512");
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "ddaf35a193617abacc417349ae20413112e6fa4e89a97ea20a9eeee64b55d39a2192992a274fc1a836ba3c23a3feebbd454d4423643ce80e2a9ac94fa54ca49f");

  mobius::crypt::hash h3 ("sha2-512");
  h3.update ( "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "8e959b75dae313da8cf4f72814fc143f8f7779c6eb9f7fa17299aeadb6889018501d289e4900f7e4331b99dec4b5433ac7d329eeb6dd26545e96e55b874be909");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("sha2-512");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("sha2-512");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_sha2_512_224 ()
{
  mobius::unittest test ("mobius::crypt::hash -> sha2-512-224");

  mobius::crypt::hash h1 ("sha2-512-224");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "6ed0dd02806fa89e25de060c19d3ac86cabb87d6a0ddd05c333b84f4");

  //! \see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA512_224.pdf
  mobius::crypt::hash h2 ("sha2-512-224");
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "4634270f707b6a54daae7530460842e20e37ed265ceee9a43e8924aa");

  mobius::crypt::hash h3 ("sha2-512-224");
  h3.update ( "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "23fec5bb94d60b23308192640b0c453335d664734fe40e7268674af9");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("sha2-512-224");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("sha2-512-224");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_sha2_512_256 ()
{
  mobius::unittest test ("mobius::crypt::hash -> sha2-512-256");

  mobius::crypt::hash h1 ("sha2-512-256");
  test.ASSERT_EQUAL (h1.get_hex_digest (), "c672b8d1ef56ed28ab87c3622c5114069bdd3ad7b8f9737498d0c01ecef0967a");

  //! \see https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA512_256.pdf
  mobius::crypt::hash h2 ("sha2-512-256");
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "53048e2681941ef99b2e29b76b4c7dabe4c2d0c634fc6d46e0e2f13107e7af23");

  mobius::crypt::hash h3 ("sha2-512-256");
  h3.update ( "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "3928e184fb8690f840da3988121d31be65cb9d3ef83ee6146feac861e19b563a");

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("sha2-512-256");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("sha2-512-256");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hash_zip ()
{
  mobius::unittest test ("mobius::crypt::hash -> zip");

  mobius::crypt::hash h1 ("zip");
  mobius::crypt::hash h2 ("zip");
  mobius::crypt::hash h3 ("zip");

  test.ASSERT_EQUAL (h1.get_hex_digest (), "123456782345678934567890");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "123456782345678934567890");
  test.ASSERT_EQUAL (h3.get_hex_digest (), "123456782345678934567890");

  // update
  h2.update ("abc");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "5dd2af4d589d03b43cf5ffa4");

  h2.update ("d");
  test.ASSERT_EQUAL (h2.get_hex_digest (), "42ef4ac38d167254428e6d93");

  h3.update ("abcd");
  test.ASSERT_EQUAL (h3.get_hex_digest (), h2.get_hex_digest ());

  // update ("a"), update ("b") = update ("ab")
  mobius::crypt::hash h100 ("zip");
  h100.update ("a");
  h100.update ("b");
  auto d100 = h100.get_digest ();

  mobius::crypt::hash h101 ("zip");
  h101.update ("ab");
  auto d101 = h101.get_digest ();

  test.ASSERT_EQUAL (h100.get_hex_digest (), h101.get_hex_digest ());

  test.end ();
}

static void
testcase_hmac ()
{
  mobius::unittest test ("mobius::crypt::hmac");

  // test data
  const struct
  {
     std::string algo;
     mobius::bytearray key;
     mobius::bytearray data;
     std::string hex_digest;
  } data[] =
  {
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    // RFC 2202 - section 2
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    {
       "md5",
       "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
       "Hi There",
       "9294727a3638bb1c13f48ef8158bfc9d"
    },
    {
       "md5",
       "Jefe",
       "what do ya want for nothing?",
       "750c783e6ab0b503eaa86e310a5db738"
    },
    {
       "md5",
       "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
       mobius::bytearray (0xdd, 50),
       "56be34521d144c88dbb8c733f0e8b3f6"
    },
    {
       "md5",
       "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19",
       mobius::bytearray (0xcd, 50),
       "697eaf0aca3a3aea3a75164746ffaa79"
    },
    {
       "md5",
       "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c",
       "Test With Truncation",
       "56461ef2342edc00f9bab995690efd4c",
    },
    {
       "md5",
       mobius::bytearray (0xaa, 80),
       "Test Using Larger Than Block-Size Key - Hash Key First",
       "6b1ab7fe4bd7bf8f0b62e6ce61b9d0cd"
    },
    {
       "md5",
       mobius::bytearray (0xaa, 80),
       "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data",
       "6f630fad67cda0ee1fb1f562db3aa53e"
    },

    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    // RFC 2202 - section 3
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    {
       "sha1",
       "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
       "Hi There",
       "b617318655057264e28bc0b6fb378c8ef146be00"
    },
    {
       "sha1",
       "Jefe",
       "what do ya want for nothing?",
       "effcdf6ae5eb2fa2d27416d5f184df9c259a7c79"
    },
    {
       "sha1",
       "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
       mobius::bytearray (0xdd, 50),
       "125d7342b9ac11cd91a39af48aa17b4f63f175d3"
    },
    {
       "sha1",
       "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19",
       mobius::bytearray (0xcd, 50),
       "4c9007f4026250c6bc8414f9bf50c86c2d7235da"
    },
    {
       "sha1",
       "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c",
       "Test With Truncation",
       "4c1a03424b55e07fe7f27be1d58bb9324a9a5a04"
    },
    {
       "sha1",
       mobius::bytearray (0xaa, 80),
       "Test Using Larger Than Block-Size Key - Hash Key First",
       "aa4ae5e15272d00e95705637ce8a3b55ed402112"
    },
    {
       "sha1",
       mobius::bytearray (0xaa, 80),
       "Test Using Larger Than Block-Size Key and Larger Than One Block-Size Data",
       "e8e99d0f45237d786d6bbaa7965c7808bbff1a91"
    },

    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    // RFC 4231 - section 4.2
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    {
       "sha2-224",
       "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
       "Hi There",
       "896fb1128abbdf196832107cd49df33f47b4b1169912ba4f53684b22"
    },
    {
       "sha2-256",
       "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
       "Hi There",
       "b0344c61d8db38535ca8afceaf0bf12b881dc200c9833da726e9376c2e32cff7"
    },
    {
       "sha2-384",
       "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
       "Hi There",
       "afd03944d84895626b0825f4ab46907f15f9dadbe4101ec682aa034c7cebc59cfaea9ea9076ede7f4af152e8b2fa9cb6"
    },
    {
       "sha2-512",
       "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
       "Hi There",
       "87aa7cdea5ef619d4ff0b4241a1d6cb02379f4e2ce4ec2787ad0b30545e17cdedaa833b7d6b8a702038b274eaea3f4e4be9d914eeb61f1702e696c203a126854"
    },

    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    // RFC 4231 - section 4.3
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    {
       "sha2-224",
       "Jefe",
       "what do ya want for nothing?",
       "a30e01098bc6dbbf45690f3a7e9e6d0f8bbea2a39e6148008fd05e44"
    },
    {
       "sha2-256",
       "Jefe",
       "what do ya want for nothing?",
       "5bdcc146bf60754e6a042426089575c75a003f089d2739839dec58b964ec3843"
    },
    {
       "sha2-384",
       "Jefe",
       "what do ya want for nothing?",
       "af45d2e376484031617f78d2b58a6b1b9c7ef464f5a01b47e42ec3736322445e8e2240ca5e69e2c78b3239ecfab21649"
    },
    {
       "sha2-512",
       "Jefe",
       "what do ya want for nothing?",
       "164b7a7bfcf819e2e395fbe73b56e0a387bd64222e831fd610270cd7ea2505549758bf75c05a994a6d034f65f8f0e6fdcaeab1a34d4a6b4b636e070a38bce737"
    },

    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    // RFC 4231 - section 4.4
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    {
       "sha2-224",
       "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
       mobius::bytearray (0xdd, 50),
       "7fb3cb3588c6c1f6ffa9694d7d6ad2649365b0c1f65d69d1ec8333ea"
    },
    {
       "sha2-256",
       "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
       mobius::bytearray (0xdd, 50),
       "773ea91e36800e46854db8ebd09181a72959098b3ef8c122d9635514ced565fe"
    },
    {
       "sha2-384",
       "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
       mobius::bytearray (0xdd, 50),
       "88062608d3e6ad8a0aa2ace014c8a86f0aa635d947ac9febe83ef4e55966144b2a5ab39dc13814b94e3ab6e101a34f27"
    },
    {
       "sha2-512",
       "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
       mobius::bytearray (0xdd, 50),
       "fa73b0089d56a284efb0f0756c890be9b1b5dbdd8ee81a3655f83e33b2279d39bf3e848279a722c806b485a47e67c807b946a337bee8942674278859e13292fb"
    },

    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    // RFC 4231 - section 4.5
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    {
       "sha2-224",
       "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19",
       mobius::bytearray (0xcd, 50),
       "6c11506874013cac6a2abc1bb382627cec6a90d86efc012de7afec5a"
    },
    {
       "sha2-256",
       "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19",
       mobius::bytearray (0xcd, 50),
       "82558a389a443c0ea4cc819899f2083a85f0faa3e578f8077a2e3ff46729665b"
    },
    {
       "sha2-384",
       "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19",
       mobius::bytearray (0xcd, 50),
       "3e8a69b7783c25851933ab6290af6ca77a9981480850009cc5577c6e1f573b4e6801dd23c4a7d679ccf8a386c674cffb"
    },
    {
       "sha2-512",
       "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19",
       mobius::bytearray (0xcd, 50),
       "b0ba465637458c6990e5a8c5f61d4af7e576d97ff94b872de76f8050361ee3dba91ca5c11aa25eb4d679275cc5788063a5f19741120c4f2de2adebeb10a298dd"
    },

    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    // RFC 4231 - section 4.7
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    {
       "sha2-224",
       mobius::bytearray (0xaa, 131),
       "Test Using Larger Than Block-Size Key - Hash Key First",
       "95e9a0db962095adaebe9b2d6f0dbce2d499f112f2d2b7273fa6870e"
    },
    {
       "sha2-256",
       mobius::bytearray (0xaa, 131),
       "Test Using Larger Than Block-Size Key - Hash Key First",
       "60e431591ee0b67f0d8a26aacbf5b77f8e0bc6213728c5140546040f0ee37f54"
    },
    {
       "sha2-384",
       mobius::bytearray (0xaa, 131),
       "Test Using Larger Than Block-Size Key - Hash Key First",
       "4ece084485813e9088d2c63a041bc5b44f9ef1012a2b588f3cd11f05033ac4c60c2ef6ab4030fe8296248df163f44952"
    },
    {
       "sha2-512",
       mobius::bytearray (0xaa, 131),
       "Test Using Larger Than Block-Size Key - Hash Key First",
       "80b24263c7c1a3ebb71493c1dd7be8b49b46d1f41b4aeec1121b013783f8f3526b56d037e05f2598bd0fd2215d6a1e5295e64f73f63f0aec8b915a985d786598"
    },

    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    // RFC 4231 - section 4.8
    // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    {
       "sha2-224",
       mobius::bytearray (0xaa, 131),
       "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
       "3a854166ac5d9f023f54d517d0b39dbd946770db9c2b95c9f6f565d1"
    },
    {
       "sha2-256",
       mobius::bytearray (0xaa, 131),
       "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
       "9b09ffa71b942fcb27635fbcd5b0e944bfdc63644f0713938a7f51535c3a35e2"
    },
    {
       "sha2-384",
       mobius::bytearray (0xaa, 131),
       "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
       "6617178e941f020d351e2f254e8fd32c602420feb0b8fb9adccebb82461e99c5a678cc31e799176d3860e6110c46523e"
    },
    {
       "sha2-512",
       mobius::bytearray (0xaa, 131),
       "This is a test using a larger than block-size key and a larger than block-size data. The key needs to be hashed before being used by the HMAC algorithm.",
       "e37b6a775dc87dbaa4dfa9f96e5e3ffddebd71f8867289865df5a32d20cdc944b6022cac3c4982b10d5eeb55c3e4de15134676fb6de0446065c97440fa8c6a58"
    },
  };

  // do tests
  for (int i = 0;i < 38;i++)
    {
      mobius::crypt::hmac h (data[i].algo, data[i].key);
      h.update (data[i].data);
      test.ASSERT_EQUAL (h.get_hex_digest (), data[i].hex_digest);
    }

  test.end ();
}

static void
testcase_pbkdf1 ()
{
  mobius::unittest test ("mobius::crypt::pbkdf1");

  //! \see http://www.di-mgt.com.au/cryptoKDFs.html
  ASSERT_EXCEPTION (test, mobius::crypt::pbkdf1 ("md5", "password", "salt", 1000, 0), std::out_of_range);

  test.ASSERT_EQUAL (mobius::crypt::pbkdf1 ("sha1", "password", "\x78\x57\x8E\x5A\x5D\x63\xCB\x06", 1000, 16).to_hexstring (), "dc19847e05c64d2faf10ebfb4a3d2a20");

  test.ASSERT_EQUAL (mobius::crypt::pbkdf1 ("md5", "password", "salt", 1000, 1).to_hexstring (), "84");
  test.ASSERT_EQUAL (mobius::crypt::pbkdf1 ("md5", "password", "salt", 1000, 8).to_hexstring (), "8475c6a8531a5d27");
  test.ASSERT_EQUAL (mobius::crypt::pbkdf1 ("md5", "password", "salt", 1000, 16).to_hexstring (), "8475c6a8531a5d27e386cd496457812c");
  test.ASSERT_EQUAL (mobius::crypt::pbkdf1 ("sha1", "password", "salt", 1000, 20).to_hexstring (), "4a8fd48e426ed081b535be5769892fa396293efb");

  test.end ();
}

static void
testcase_pbkdf2_hmac ()
{
  mobius::unittest test ("mobius::crypt::pbkdf2_hmac");

  // RFC 6070 - section 2
  ASSERT_EXCEPTION (test, mobius::crypt::pbkdf2_hmac ("md5", "password", "salt", 1000, 0), std::out_of_range);

  test.ASSERT_EQUAL (mobius::crypt::pbkdf2_hmac ("sha1", "password", "salt", 1, 20).to_hexstring (), "0c60c80f961f0e71f3a9b524af6012062fe037a6");
  test.ASSERT_EQUAL (mobius::crypt::pbkdf2_hmac ("sha1", "password", "salt", 2, 20).to_hexstring (), "ea6c014dc72d6f8ccd1ed92ace1d41f0d8de8957");
  test.ASSERT_EQUAL (mobius::crypt::pbkdf2_hmac ("sha1", "password", "salt", 4096, 20).to_hexstring (),"4b007901b765489abead49d926f721d065a429c1");
  //test.ASSERT_EQUAL (pbkdf2_hmac ("password", "salt", 16777216, 20, "sha1").to_hexstring (),"eefe3d61cd4da4e4e9945b3d6ba2158c2634e984"); .. very slow
  test.ASSERT_EQUAL (mobius::crypt::pbkdf2_hmac ("sha1", "passwordPASSWORDpassword", "saltSALTsaltSALTsaltSALTsaltSALTsalt", 4096, 25).to_hexstring (), "3d2eec4fe41c849b80c8d83662c0e44a8b291a964cf2f07038");
  test.ASSERT_EQUAL (mobius::crypt::pbkdf2_hmac ("sha1", {'p','a','s','s',0,'w','o','r','d'}, {'s','a',0,'l','t'}, 4096, 16).to_hexstring (), "56fa6aa75548099dcc37d7f03425e0c3");

  // RFC 7914 - section 11
  test.ASSERT_EQUAL (mobius::crypt::pbkdf2_hmac ("sha2-256", "passwd", "salt", 1, 64).to_hexstring (), "55ac046e56e3089fec1691c22544b605f94185216dde0465e68b9d57c20dacbc49ca9cccf179b645991664b39d77ef317c71b845b1e30bd509112041d3a19783");
  test.ASSERT_EQUAL (mobius::crypt::pbkdf2_hmac ("sha2-256", "Password", "NaCl", 80000, 64).to_hexstring (), "4ddcd8f60b98be21830cee5ef22701f9641a4418d04c0414aeff08876b34ab56a1d425a1225833549adb841b51c9b3176a272bdebba1d078478f62b397f33c8d");

  test.end ();
}

static void
testcase_rot13 ()
{
  mobius::unittest test ("mobius::crypt::rot13");

  // nowhere <-> abjurer
  test.ASSERT_EQUAL (mobius::crypt::rot13 ("nowhere"), "abjurer");
  test.ASSERT_EQUAL (mobius::crypt::rot13 ("abjurer"), "nowhere");

  // NERF <-> ARES
  test.ASSERT_EQUAL (mobius::crypt::rot13 ("ARES"), "NERF");
  test.ASSERT_EQUAL (mobius::crypt::rot13 ("NERF"), "ARES");

  // Numbers
  test.ASSERT_EQUAL (mobius::crypt::rot13 ("0123456789"), "0123456789");

  test.end ();
}

void
unittest_crypt ()
{
  testcase_cipher_3des ();
  testcase_cipher_aes ();
  testcase_cipher_blowfish ();
  testcase_cipher_des ();
  testcase_cipher_rc2 ();
  testcase_cipher_rc4 ();
  testcase_cipher_zip ();
  testcase_hash_adler32 ();
  testcase_hash_crc32 ();
  testcase_hash_ed2k ();
  testcase_hash_md4 ();
  testcase_hash_md5 ();
  testcase_hash_sha1 ();
  testcase_hash_sha2_224 ();
  testcase_hash_sha2_256 ();
  testcase_hash_sha2_384 ();
  testcase_hash_sha2_512 ();
  testcase_hash_sha2_512_224 ();
  testcase_hash_sha2_512_256 ();
  testcase_hash_zip ();
  testcase_hmac ();
  testcase_pbkdf1 ();
  testcase_pbkdf2_hmac ();
  testcase_rot13 ();
}
