
/*
 * attenuation_coefficient.cc
 *
 *  Copyright (C) 2013 Diamond Light Source
 *
 *  Author: James Parkhurst
 *
 *  This code is distributed under the BSD license, a copy of which is
 *  included in the root directory of this package.
 */

#include <cctbx/eltbx/attenuation_coefficient.h>

namespace cctbx { namespace eltbx { namespace attenuation_coefficient {

  const double DENSITY[XRAY_MASS_COEFF_TABLE_SIZE] = {
    0.0, 8.375e-05, 0.0001663, 0.534, 1.848, 2.37, 1.7, 0.001165, 0.001332,
    0.00158, 0.0008385, 0.971, 1.74, 2.699, 2.33, 2.2, 2.0, 0.002995, 0.001662,
    0.862, 1.55, 2.989, 4.54, 6.11, 7.18, 7.44, 7.874, 8.9, 8.902, 8.96, 7.133,
    5.904, 5.323, 5.73, 4.5, 0.007072, 0.003478, 1.532, 2.54, 4.469, 6.506,
    8.57, 10.22, 11.5, 12.41, 12.41, 12.02, 10.5, 8.65, 7.31, 7.31, 6.691, 6.24,
    4.93, 0.005485, 1.873, 3.5, 6.154, 6.657, 6.71, 6.9, 7.22, 7.46, 5.243, 7.9,
    8.229, 8.55, 8.795, 9.066, 9.321, 6.73, 9.84, 13.31, 16.65, 19.3, 21.02,
    22.57, 22.42, 21.45, 19.32, 13.55, 11.72, 11.35, 9.747, 9.32, 10.0,
    0.009066, 10.0, 5.0, 10.07, 11.72, 15.37, 18.95,
    // composite materials
    6.2, 5.32,
    };

  const table_data XRAY_MASS_COEFF_TABLE[XRAY_MASS_COEFF_TABLE_SIZE] = {
    { 0, {}, {}, {} },

    // Format:
    //   Number of entries at different energy levels,
    //   Energy (MeV),
    //   mu    / rho (cm^2 / g),
    //   mu_en / rho (cm^2 / g)

    // Z = 1
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        7.217, 2.148, 1.059, 0.5612, 0.4546, 0.4193, 0.4042, 0.3914, 0.3854,
        0.3764, 0.3695, 0.357, 0.3458, 0.3355, 0.326, 0.3091, 0.2944,
        0.2651, 0.2429, 0.2112, 0.1893, 0.1729, 0.1599, 0.1405, 0.1263,
        0.1129, 0.1027, 0.08769, 0.06921, 0.05806, 0.05049, 0.04498,
        0.03746, 0.03254, 0.02539, 0.02153,
      }, {
        6.82, 1.752, 0.6643, 0.1693, 0.06549, 0.03278, 0.01996, 0.0116,
        0.009849, 0.01102, 0.01355, 0.01863, 0.02315, 0.02709, 0.03053,
        0.0362, 0.04063, 0.04813, 0.05254, 0.05695, 0.0586, 0.059,
        0.05875, 0.05739, 0.05556, 0.05311, 0.05075, 0.0465, 0.03992,
        0.03523, 0.03174, 0.02905, 0.02515, 0.02247, 0.01837, 0.01606,
      }
    },


    // Z = 2
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        60.84, 16.76, 6.863, 2.007, 0.9329, 0.5766, 0.4195, 0.2933, 0.2476,
        0.2092, 0.196, 0.1838, 0.1763, 0.1703, 0.1651, 0.1562, 0.1486,
        0.1336, 0.1224, 0.1064, 0.09535, 0.08707, 0.08054, 0.07076,
        0.06362, 0.05688, 0.05173, 0.04422, 0.03503, 0.02949, 0.02577,
        0.02307, 0.0194, 0.01703, 0.01363, 0.01183,
      }, {
        60.45, 16.38, 6.503, 1.681, 0.6379, 0.3061, 0.1671, 0.06446, 0.0326,
        0.01246, 0.00941, 0.01003, 0.0119, 0.01375, 0.01544, 0.01826,
        0.02047, 0.02424, 0.02647, 0.02868, 0.02951, 0.02971, 0.02959,
        0.0289, 0.02797, 0.02674, 0.02555, 0.02343, 0.02019, 0.0179,
        0.01622, 0.01493, 0.01308, 0.01183, 0.009948, 0.008914,
      }
    },


    // Z = 3
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        233.9, 66.68, 27.07, 7.549, 3.114, 1.619, 0.9875, 0.5054, 0.3395,
        0.2176, 0.1856, 0.1644, 0.1551, 0.1488, 0.1438, 0.1356, 0.1289,
        0.1158, 0.106, 0.0921, 0.08249, 0.07532, 0.06968, 0.06121,
        0.05503, 0.04921, 0.04476, 0.0383, 0.03043, 0.02572, 0.02257,
        0.0203, 0.01725, 0.01529, 0.01252, 0.01109,
      }, {
        233.5, 66.29, 26.72, 7.249, 2.84, 1.364, 0.7477, 0.2888, 0.1387,
        0.0391, 0.01885, 0.01138, 0.01131, 0.01237, 0.01361, 0.01588,
        0.01776, 0.02098, 0.0229, 0.02481, 0.02552, 0.02569, 0.02559,
        0.02499, 0.02419, 0.02312, 0.0221, 0.02028, 0.01753, 0.01561,
        0.01422, 0.01316, 0.01167, 0.01066, 0.009182, 0.008385,
      }
    },


    // Z = 4
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        604.1, 179.7, 74.69, 21.27, 8.685, 4.369, 2.527, 1.124, 0.6466, 0.307,
        0.2251, 0.1792, 0.164, 0.1554, 0.1493, 0.1401, 0.1328, 0.119,
        0.1089, 0.09463, 0.08471, 0.07739, 0.07155, 0.06286, 0.05652,
        0.05054, 0.04597, 0.03938, 0.03138, 0.02664, 0.02347, 0.02121,
        0.01819, 0.01627, 0.01361, 0.01227,
      }, {
        603.5, 179.1, 74.22, 20.9, 8.367, 4.081, 2.26, 0.8839, 0.4255, 0.1143,
        0.0478, 0.01898, 0.01438, 0.01401, 0.01468, 0.01658, 0.01836,
        0.02157, 0.02353, 0.02548, 0.0262, 0.02639, 0.02627, 0.02565,
        0.02483, 0.02373, 0.02268, 0.02083, 0.01806, 0.01617, 0.01479,
        0.01377, 0.01233, 0.01138, 0.01001, 0.009294,
      }
    },


    // Z = 5
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        1229.0, 376.6, 159.7, 46.67, 19.27, 9.683, 5.538, 2.346, 1.255,
        0.4827, 0.3014, 0.2063, 0.1793, 0.1665, 0.1583, 0.1472, 0.1391,
        0.1243, 0.1136, 0.09862, 0.08834, 0.08065, 0.0746, 0.06549,
        0.0589, 0.05266, 0.04791, 0.04108, 0.03284, 0.02798, 0.02476,
        0.02248, 0.01945, 0.01755, 0.01495, 0.01368,
      }, {
        1228.0, 375.9, 159.1, 46.17, 18.86, 9.332, 5.223, 2.072, 1.006,
        0.2698, 0.1084, 0.03506, 0.02084, 0.01737, 0.0168, 0.01785,
        0.0194, 0.02255, 0.02453, 0.02654, 0.02731, 0.02749, 0.02737,
        0.02671, 0.02586, 0.02472, 0.02362, 0.02171, 0.01889, 0.01698,
        0.01562, 0.01461, 0.01322, 0.01232, 0.01104, 0.01039,
      }
    },


    // Z = 6
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        2211.0, 700.2, 302.6, 90.33, 37.78, 19.12, 10.95, 4.576, 2.373,
        0.8071, 0.442, 0.2562, 0.2076, 0.1871, 0.1753, 0.161, 0.1514,
        0.1347, 0.1229, 0.1066, 0.09546, 0.08715, 0.08058, 0.07076,
        0.06361, 0.0569, 0.05179, 0.04442, 0.03562, 0.03047, 0.02708,
        0.02469, 0.02154, 0.01959, 0.01698, 0.01575,
      }, {
        2209.0, 699.0, 301.6, 89.63, 37.23, 18.66, 10.54, 4.242, 2.078,
        0.5627, 0.2238, 0.06614, 0.03343, 0.02397, 0.02098, 0.02037,
        0.02147, 0.02449, 0.02655, 0.0287, 0.0295, 0.02969, 0.02956,
        0.02885, 0.02792, 0.02669, 0.02551, 0.02345, 0.02048, 0.01849,
        0.0171, 0.01607, 0.01468, 0.0138, 0.01258, 0.01198,
      }
    },


    // Z = 7
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        3311.0, 1083.0, 476.9, 145.6, 61.66, 31.44, 18.09, 7.562, 3.879,
        1.236, 0.6178, 0.3066, 0.2288, 0.198, 0.1817, 0.1639, 0.1529,
        0.1353, 0.1233, 0.1068, 0.09557, 0.08719, 0.08063, 0.07081,
        0.06364, 0.05693, 0.0518, 0.0445, 0.03579, 0.03073, 0.02742,
        0.02511, 0.02209, 0.02024, 0.01782, 0.01673,
      }, {
        3306.0, 1080.0, 475.5, 144.7, 60.94, 30.86, 17.59, 7.17, 3.545,
        0.9715, 0.3867, 0.1099, 0.05051, 0.03217, 0.02548, 0.02211,
        0.02231, 0.02472, 0.02665, 0.02873, 0.02952, 0.02969, 0.02956,
        0.02886, 0.02792, 0.02669, 0.0255, 0.02347, 0.02057, 0.01867,
        0.01734, 0.01639, 0.01512, 0.01434, 0.01332, 0.01285,
      }
    },


    // Z = 8
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        4590.0, 1549.0, 694.9, 217.1, 93.15, 47.9, 27.7, 11.63, 5.952, 1.836,
        0.8651, 0.3779, 0.2585, 0.2132, 0.1907, 0.1678, 0.1551, 0.1361,
        0.1237, 0.107, 0.09566, 0.08729, 0.0807, 0.07087, 0.06372,
        0.05697, 0.05185, 0.04459, 0.03597, 0.031, 0.02777, 0.02552,
        0.02263, 0.02089, 0.01866, 0.0177,
      }, {
        4576.0, 1545.0, 692.6, 215.8, 92.21, 47.15, 27.08, 11.16, 5.565,
        1.545, 0.6179, 0.1729, 0.0753, 0.04414, 0.03207, 0.02468,
        0.02355, 0.02506, 0.02679, 0.02877, 0.02953, 0.02971, 0.02957,
        0.02887, 0.02794, 0.02669, 0.02551, 0.0235, 0.02066, 0.01882,
        0.01757, 0.01668, 0.01553, 0.01483, 0.01396, 0.0136,
      }
    },


    // Z = 9
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        5649.0, 1979.0, 904.7, 288.8, 125.6, 65.14, 37.89, 16.02, 8.205,
        2.492, 1.133, 0.4487, 0.2828, 0.2214, 0.192, 0.1639, 0.1496,
        0.1298, 0.1176, 0.1015, 0.09073, 0.08274, 0.07649, 0.06717,
        0.06037, 0.05399, 0.04915, 0.04228, 0.03422, 0.0296, 0.02663,
        0.02457, 0.02195, 0.02039, 0.01846, 0.01769,
      }, {
        5615.0, 1969.0, 900.5, 287.0, 124.4, 64.24, 37.16, 15.48, 7.776,
        2.186, 0.8796, 0.2451, 0.1036, 0.05747, 0.03903, 0.02676,
        0.02394, 0.02417, 0.02554, 0.02729, 0.028, 0.02815, 0.02801,
        0.02734, 0.02645, 0.02527, 0.02416, 0.02226, 0.01964, 0.01797,
        0.01685, 0.01607, 0.01508, 0.01451, 0.01382, 0.01357,
      }
    },


    // Z = 10
    {
      36, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4,
        0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0,
        10.0, 15.0, 20.0,
      }, {
        7409.0, 2666.0, 1243.0, 405.1, 178.5, 93.39, 54.67, 23.28, 11.97,
        3.613, 1.606, 0.5923, 0.3473, 0.2579, 0.2161, 0.1781, 0.16,
        0.137, 0.1236, 0.1064, 0.09502, 0.08664, 0.08006, 0.07029,
        0.06316, 0.05646, 0.05145, 0.0443, 0.03594, 0.03122, 0.02818,
        0.0261, 0.02348, 0.02197, 0.02013, 0.01946,
      }, {
        7326.0, 2645.0, 1234.0, 402.1, 176.7, 92.14, 53.69, 22.6, 11.43,
        3.253, 1.317, 0.3676, 0.1528, 0.08182, 0.05287, 0.03273,
        0.02733, 0.0259, 0.02697, 0.02862, 0.02931, 0.02946, 0.0293,
        0.0286, 0.02766, 0.02641, 0.02526, 0.0233, 0.02061, 0.01894,
        0.01784, 0.01709, 0.01616, 0.01564, 0.01508, 0.01491,
      }
    },


    // Z = 11
    {
      39, {
        0.001, 0.00103542, 0.0010721, 0.0010721, 0.0015, 0.002, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06,
        0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5,
        2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        654.2, 596.0, 542.9, 6435.0, 3194.0, 1521.0, 507.0, 226.1, 119.4,
        70.3, 30.18, 15.57, 4.694, 2.057, 0.7197, 0.3969, 0.2804,
        0.2268, 0.1796, 0.1585, 0.1335, 0.1199, 0.1029, 0.09185,
        0.08372, 0.07736, 0.06788, 0.061, 0.05454, 0.04968, 0.04282,
        0.03487, 0.03037, 0.02753, 0.02559, 0.02319, 0.02181, 0.02023,
        0.0197,
      }, {
        652.2, 594.1, 541.0, 6320.0, 3151.0, 1504.0, 502.3, 223.8, 117.8,
        69.15, 29.41, 14.99, 4.313, 1.759, 0.4928, 0.2031, 0.1063,
        0.06625, 0.03761, 0.02931, 0.02579, 0.02635, 0.02771, 0.02833,
        0.02845, 0.0283, 0.0276, 0.02669, 0.02549, 0.02437, 0.02249,
        0.01997, 0.01842, 0.01743, 0.01675, 0.01595, 0.01552, 0.01508,
        0.01496,
      }
    },


    // Z = 12
    {
      39, {
        0.001, 0.00114237, 0.001305, 0.001305, 0.0015, 0.002, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06,
        0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5,
        2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        922.5, 647.4, 453.0, 5444.0, 4004.0, 1932.0, 658.5, 297.4, 158.3,
        93.81, 40.61, 21.05, 6.358, 2.763, 0.9306, 0.4881, 0.3292,
        0.257, 0.1951, 0.1686, 0.1394, 0.1245, 0.1065, 0.09492, 0.08647,
        0.07988, 0.07008, 0.06296, 0.05629, 0.05129, 0.04426, 0.03613,
        0.03159, 0.02873, 0.02681, 0.02445, 0.02313, 0.02168, 0.02127,
      }, {
        920.3, 645.2, 450.9, 5310.0, 3918.0, 1899.0, 649.9, 293.7, 156.1,
        92.27, 39.65, 20.36, 5.925, 2.432, 0.6855, 0.2815, 0.1451,
        0.0882, 0.04671, 0.0341, 0.02766, 0.02761, 0.02871, 0.02928,
        0.02938, 0.02921, 0.02848, 0.02753, 0.02629, 0.02514, 0.02322,
        0.02067, 0.01915, 0.01819, 0.01756, 0.01683, 0.01646, 0.01614,
        0.01609,
      }
    },


    // Z = 13
    {
      38, {
        0.001, 0.0015, 0.0015596, 0.0015596, 0.002, 0.003, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        1185.0, 402.2, 362.1, 3957.0, 2263.0, 788.0, 360.5, 193.4, 115.3,
        50.33, 26.23, 7.955, 3.441, 1.128, 0.5685, 0.3681, 0.2778,
        0.2018, 0.1704, 0.1378, 0.1223, 0.1042, 0.09276, 0.08445,
        0.07802, 0.06841, 0.06146, 0.05496, 0.05006, 0.04324, 0.03541,
        0.03106, 0.02836, 0.02655, 0.02437, 0.02318, 0.02195, 0.02168,
      }, {
        1183.0, 400.1, 360.0, 3829.0, 2204.0, 773.2, 354.5, 190.2, 113.3,
        49.18, 25.43, 7.487, 3.094, 0.8778, 0.3601, 0.184, 0.1099,
        0.05511, 0.03794, 0.02827, 0.02745, 0.02816, 0.02862, 0.02868,
        0.02851, 0.02778, 0.02686, 0.02565, 0.02451, 0.02266, 0.02024,
        0.01882, 0.01795, 0.01739, 0.01678, 0.0165, 0.01631, 0.01633,
      }
    },


    // Z = 14 Si
    {
      38, {
        0.001, 0.0015, 0.0018389, 0.0018389, 0.002, 0.003, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        1570.0, 535.5, 309.2, 3192.0, 2777.0, 978.4, 452.9, 245.0, 147.0,
        64.68, 33.89, 10.34, 4.464, 1.436, 0.7012, 0.4385, 0.3207,
        0.2228, 0.1835, 0.1448, 0.1275, 0.1082, 0.09614, 0.08748,
        0.08077, 0.07082, 0.06361, 0.05688, 0.05183, 0.0448, 0.03678,
        0.0324, 0.02967, 0.02788, 0.02574, 0.02462, 0.02352, 0.02338,
      }, {
        1567.0, 533.1, 307.0, 3059.0, 2669.0, 951.6, 442.7, 240.0, 143.9,
        63.13, 32.89, 9.794, 4.076, 1.164, 0.4782, 0.243, 0.1434,
        0.06896, 0.04513, 0.03086, 0.02905, 0.02932, 0.02968, 0.02971,
        0.02951, 0.02875, 0.02778, 0.02652, 0.02535, 0.02345, 0.02101,
        0.01963, 0.01878, 0.01827, 0.01773, 0.01753, 0.01746, 0.01757,
      }
    },


    // Z = 15
    {
      38, {
        0.001, 0.0015, 0.002, 0.0021455, 0.0021455, 0.003, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        1913.0, 654.7, 301.8, 249.4, 2473.0, 1118.0, 524.2, 286.0, 172.6,
        76.6, 40.35, 12.39, 5.352, 1.7, 0.8096, 0.4916, 0.3494, 0.2324,
        0.1865, 0.1432, 0.125, 0.1055, 0.09359, 0.08511, 0.07854,
        0.06884, 0.06182, 0.05526, 0.05039, 0.04358, 0.0359, 0.03172,
        0.02915, 0.02747, 0.02552, 0.02452, 0.02364, 0.02363,
      }, {
        1910.0, 652.2, 299.6, 247.3, 2343.0, 1074.0, 507.9, 278.2, 168.2,
        74.57, 39.12, 11.79, 4.939, 1.422, 0.585, 0.2965, 0.1735,
        0.08083, 0.05068, 0.03188, 0.02899, 0.0287, 0.02892, 0.02891,
        0.02869, 0.02793, 0.02698, 0.02575, 0.02462, 0.02279, 0.02049,
        0.01921, 0.01846, 0.01801, 0.0176, 0.01747, 0.01754, 0.01771,
      }
    },


    // Z = 16
    {
      38, {
        0.001, 0.0015, 0.002, 0.002472, 0.002472, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1,
        0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0,
        4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2429.0, 834.2, 385.3, 216.8, 2070.0, 1339.0, 633.8, 348.7, 211.6,
        94.65, 50.12, 15.5, 6.708, 2.113, 0.9872, 0.5849, 0.4053,
        0.2585, 0.202, 0.1506, 0.1302, 0.1091, 0.09665, 0.08781,
        0.08102, 0.07098, 0.06373, 0.05697, 0.05193, 0.04498, 0.03715,
        0.03293, 0.03036, 0.02872, 0.02682, 0.02589, 0.02517, 0.02529,
      }, {
        2426.0, 831.4, 382.8, 214.5, 1935.0, 1265.0, 606.6, 336.0, 204.6,
        91.71, 48.47, 14.77, 6.235, 1.809, 0.7466, 0.3779, 0.2199, 0.1,
        0.06052, 0.03516, 0.0308, 0.02983, 0.02991, 0.02984, 0.02959,
        0.02878, 0.0278, 0.02652, 0.02535, 0.02349, 0.02118, 0.01993,
        0.01923, 0.01884, 0.0185, 0.01845, 0.01864, 0.01889,
      }
    },


    // Z = 17
    {
      38, {
        0.001, 0.0015, 0.002, 0.0028224, 0.0028224, 0.003, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2832.0, 977.1, 452.0, 177.4, 1637.0, 1473.0, 703.7, 390.1, 238.4,
        107.5, 57.25, 17.84, 7.739, 2.425, 1.117, 0.6483, 0.4395,
        0.2696, 0.205, 0.148, 0.1266, 0.1054, 0.09311, 0.08453, 0.07795,
        0.06826, 0.06128, 0.05478, 0.04994, 0.04328, 0.03585, 0.03188,
        0.0295, 0.02798, 0.02628, 0.02549, 0.02496, 0.0252,
      }, {
        2829.0, 974.2, 449.4, 175.2, 1506.0, 1361.0, 662.6, 371.1, 228.2,
        103.4, 55.1, 17.0, 7.227, 2.114, 0.8756, 0.4433, 0.257, 0.1148,
        0.06745, 0.03639, 0.03067, 0.02898, 0.02887, 0.02874, 0.02847,
        0.02767, 0.02671, 0.02549, 0.02436, 0.02258, 0.02043, 0.01931,
        0.01871, 0.01839, 0.01819, 0.01824, 0.01861, 0.01899,
      }
    },


    // Z = 18
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.0032029, 0.0032029, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        3184.0, 1105.0, 512.0, 170.3, 142.4, 1275.0, 757.2, 422.5, 259.3,
        118.0, 63.16, 19.83, 8.629, 2.697, 1.228, 0.7012, 0.4664, 0.276,
        0.2043, 0.1427, 0.1205, 0.09953, 0.08776, 0.07958, 0.07335,
        0.06419, 0.05762, 0.0515, 0.04695, 0.04074, 0.03384, 0.03019,
        0.02802, 0.02667, 0.02517, 0.02451, 0.02418, 0.02453,
      }, {
        3180.0, 1102.0, 509.3, 168.2, 140.3, 1153.0, 697.9, 395.3, 244.9,
        112.5, 60.38, 18.86, 8.074, 2.382, 0.9907, 0.502, 0.2904, 0.128,
        0.07344, 0.03703, 0.02998, 0.02757, 0.02727, 0.02708, 0.02679,
        0.02601, 0.0251, 0.02394, 0.02288, 0.02123, 0.01927, 0.01827,
        0.01777, 0.01753, 0.01742, 0.01754, 0.018, 0.01842,
      }
    },


    // Z = 19
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.0036074, 0.0036074, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        4058.0, 1418.0, 659.2, 219.8, 132.7, 1201.0, 925.6, 518.9, 320.5,
        146.9, 79.07, 25.03, 10.93, 3.413, 1.541, 0.8679, 0.5678,
        0.3251, 0.2345, 0.1582, 0.1319, 0.108, 0.09495, 0.086, 0.07922,
        0.06929, 0.06216, 0.05556, 0.05068, 0.04399, 0.03666, 0.03282,
        0.03054, 0.02915, 0.02766, 0.02704, 0.02687, 0.02737,
      }, {
        4053.0, 1415.0, 656.3, 217.4, 130.6, 1065.0, 830.4, 475.2, 297.4,
        138.3, 74.9, 23.7, 10.23, 3.045, 1.272, 0.6454, 0.373, 0.1628,
        0.09161, 0.04346, 0.03378, 0.03015, 0.02959, 0.02929, 0.02895,
        0.02807, 0.02707, 0.02581, 0.02467, 0.0229, 0.02084, 0.01985,
        0.01935, 0.01914, 0.0191, 0.01928, 0.01983, 0.02029,
      }
    },


    // Z = 20
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.0040381, 0.0040381, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        4867.0, 1714.0, 799.9, 267.6, 121.8, 118.7, 1023.0, 602.6, 373.1,
        172.6, 93.41, 29.79, 13.06, 4.08, 1.83, 1.019, 0.6578, 0.3656,
        0.2571, 0.1674, 0.1376, 0.1116, 0.09783, 0.08851, 0.08148,
        0.07122, 0.06388, 0.05709, 0.05207, 0.04524, 0.0378, 0.03395,
        0.0317, 0.03035, 0.02892, 0.02839, 0.02838, 0.02903,
      }, {
        4861.0, 1710.0, 796.6, 265.0, 119.7, 116.6, 888.7, 537.3, 338.7,
        160.0, 87.44, 28.04, 12.2, 3.665, 1.538, 0.7822, 0.452, 0.1958,
        0.1085, 0.04876, 0.03639, 0.03146, 0.0306, 0.03019, 0.02979,
        0.02884, 0.0278, 0.0265, 0.02532, 0.02352, 0.02147, 0.02052,
        0.02007, 0.01992, 0.01997, 0.02022, 0.02088, 0.0214,
      }
    },


    // Z = 21
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.0044928, 0.0044928, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        5238.0, 1858.0, 870.6, 292.2, 133.2, 96.87, 814.8, 630.5, 393.3,
        182.8, 99.52, 32.02, 14.09, 4.409, 1.969, 1.087, 0.6932, 0.3753,
        0.2577, 0.1619, 0.131, 0.1052, 0.09193, 0.08305, 0.07639,
        0.06675, 0.05985, 0.05347, 0.04878, 0.04243, 0.03554, 0.03202,
        0.02999, 0.02878, 0.02756, 0.02715, 0.02732, 0.02804,
      }, {
        5231.0, 1853.0, 867.2, 289.6, 131.1, 94.9, 691.1, 543.6, 347.4, 166.1,
        91.75, 29.88, 13.11, 3.975, 1.677, 0.8552, 0.4944, 0.2132,
        0.1167, 0.04998, 0.03586, 0.02997, 0.02887, 0.02837, 0.02795,
        0.02703, 0.02603, 0.0248, 0.0237, 0.02202, 0.02016, 0.01933,
        0.01897, 0.01887, 0.019, 0.01929, 0.02, 0.02052,
      }
    },


    // Z = 22
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.0049664, 0.0049664, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        5869.0, 2096.0, 986.0, 332.3, 151.7, 83.8, 687.8, 683.8, 432.3, 202.3,
        110.7, 35.87, 15.85, 4.972, 2.214, 1.213, 0.7661, 0.4052,
        0.2721, 0.1649, 0.1314, 0.1043, 0.09081, 0.08191, 0.07529,
        0.06572, 0.05891, 0.05263, 0.04801, 0.0418, 0.03512, 0.03173,
        0.02982, 0.02868, 0.02759, 0.02727, 0.02762, 0.02844,
      }, {
        5860.0, 2091.0, 982.4, 329.5, 149.4, 81.88, 568.4, 565.7, 369.1,
        179.3, 100.1, 33.11, 14.65, 4.488, 1.904, 0.9737, 0.5634,
        0.2422, 0.1312, 0.05393, 0.03726, 0.03007, 0.02864, 0.02804,
        0.02756, 0.02661, 0.02561, 0.02439, 0.0233, 0.02166, 0.01989,
        0.01913, 0.01884, 0.01879, 0.01899, 0.01933, 0.02013, 0.02067,
      }
    },


    // Z = 23
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.0054651, 0.0054651,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        6495.0, 2342.0, 1106.0, 374.3, 171.2, 92.91, 72.77, 587.0, 468.7,
        221.7, 121.8, 39.83, 17.68, 5.564, 2.472, 1.347, 0.8438, 0.4371,
        0.2877, 0.1682, 0.1318, 0.1034, 0.08965, 0.08074, 0.07414,
        0.06466, 0.05794, 0.05175, 0.04722, 0.04115, 0.03466, 0.03141,
        0.0296, 0.02855, 0.02759, 0.02738, 0.02786, 0.02877,
      }, {
        6483.0, 2336.0, 1102.0, 371.3, 168.8, 90.89, 70.9, 471.7, 384.1,
        190.8, 107.7, 36.27, 16.2, 5.015, 2.14, 1.098, 0.6364, 0.2731,
        0.1469, 0.05821, 0.03879, 0.03019, 0.02843, 0.0277, 0.02717,
        0.02618, 0.02518, 0.02396, 0.02289, 0.0213, 0.01961, 0.01892,
        0.01869, 0.01869, 0.01896, 0.01937, 0.02022, 0.02079,
      }
    },


    // Z = 24
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.0059892, 0.0059892,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        7405.0, 2694.0, 1277.0, 433.9, 198.8, 108.0, 65.74, 597.7, 516.0,
        251.3, 138.6, 45.71, 20.38, 6.434, 2.856, 1.55, 0.9639, 0.4905,
        0.3166, 0.1788, 0.1378, 0.1067, 0.09213, 0.08281, 0.07598,
        0.0662, 0.0593, 0.05295, 0.04832, 0.04213, 0.03559, 0.03235,
        0.03057, 0.02956, 0.02869, 0.02855, 0.0292, 0.03026,
      }, {
        7388.0, 2687.0, 1272.0, 430.5, 196.1, 105.7, 63.83, 402.7, 402.7,
        208.7, 119.3, 40.93, 18.46, 5.78, 2.482, 1.278, 0.742, 0.3182,
        0.1701, 0.06536, 0.04211, 0.0316, 0.02938, 0.02849, 0.02788,
        0.0268, 0.02576, 0.0245, 0.0234, 0.02178, 0.02011, 0.01947,
        0.01929, 0.01933, 0.0197, 0.02016, 0.02112, 0.02174,
      }
    },


    // Z = 25
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.006539, 0.006539,
        0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1,
        0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0,
        4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        8093.0, 2984.0, 1421.0, 485.1, 222.9, 121.2, 73.5, 58.03, 452.0,
        273.4, 151.4, 50.27, 22.53, 7.141, 3.169, 1.714, 1.06, 0.5306,
        0.3367, 0.1838, 0.1391, 0.1062, 0.09133, 0.08192, 0.07509,
        0.06537, 0.05852, 0.05224, 0.04769, 0.04162, 0.03524, 0.03213,
        0.03045, 0.02952, 0.02875, 0.02871, 0.02951, 0.03068,
      }, {
        8069.0, 2975.0, 1415.0, 481.5, 220.1, 118.8, 71.5, 56.19, 341.5,
        217.7, 126.2, 44.14, 20.15, 6.382, 2.757, 1.425, 0.8294, 0.3558,
        0.1894, 0.07085, 0.04421, 0.03196, 0.02932, 0.02827, 0.0276,
        0.02647, 0.02541, 0.02416, 0.02307, 0.02149, 0.01989, 0.01933,
        0.0192, 0.0193, 0.01973, 0.02025, 0.02127, 0.02193,
      }
    },


    // Z = 26
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.007112, 0.007112,
        0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1,
        0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0,
        4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        9085.0, 3399.0, 1626.0, 557.6, 256.7, 139.8, 84.84, 53.19, 407.6,
        305.6, 170.6, 57.08, 25.68, 8.176, 3.629, 1.958, 1.205, 0.5952,
        0.3717, 0.1964, 0.146, 0.1099, 0.094, 0.08414, 0.07704, 0.06699,
        0.05995, 0.0535, 0.04883, 0.04265, 0.03621, 0.03312, 0.03146,
        0.03057, 0.02991, 0.02994, 0.03092, 0.03224,
      }, {
        9052.0, 3388.0, 1620.0, 553.5, 253.6, 137.2, 82.65, 51.33, 297.8,
        231.6, 136.9, 48.96, 22.6, 7.251, 3.155, 1.638, 0.9555, 0.4104,
        0.2177, 0.07961, 0.04825, 0.03361, 0.03039, 0.02914, 0.02836,
        0.02714, 0.02603, 0.02472, 0.0236, 0.02199, 0.02042, 0.0199,
        0.01983, 0.01997, 0.0205, 0.02108, 0.02221, 0.02292,
      }
    },


    // Z = 27
    {
      38, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.006, 0.0077089,
        0.0077089, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06,
        0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5,
        2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        9796.0, 3697.0, 1779.0, 612.9, 283.0, 154.3, 93.7, 47.1, 355.5, 324.8,
        184.1, 62.01, 28.03, 8.962, 3.981, 2.144, 1.314, 0.6414, 0.3949,
        0.2023, 0.1476, 0.1094, 0.09311, 0.08315, 0.07604, 0.06604,
        0.05906, 0.0527, 0.0481, 0.04204, 0.0358, 0.03283, 0.03127,
        0.03045, 0.02991, 0.03002, 0.03115, 0.03256,
      }, {
        9750.0, 3682.0, 1771.0, 608.4, 279.6, 151.5, 91.38, 45.3, 250.7,
        232.2, 141.3, 51.78, 24.21, 7.873, 3.45, 1.799, 1.052, 0.4528,
        0.2397, 0.08597, 0.05071, 0.03406, 0.03034, 0.0289, 0.02805,
        0.02677, 0.02564, 0.02434, 0.02323, 0.02165, 0.02016, 0.01971,
        0.01969, 0.01988, 0.02048, 0.0211, 0.0223, 0.02303,
      }
    },


    // Z = 28
    {
      41, {
        0.001, 0.00100404, 0.0010081, 0.0010081, 0.0015, 0.002, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.0083328, 0.0083328, 0.01, 0.015, 0.02,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        9855.0, 9753.0, 9654.0, 10990.0, 4234.0, 2049.0, 709.4, 328.2, 179.3,
        109.0, 49.52, 44.28, 329.4, 209.0, 70.81, 32.2, 10.34, 4.6,
        2.474, 1.512, 0.7306, 0.444, 0.2208, 0.1582, 0.1154, 0.09765,
        0.08698, 0.07944, 0.06891, 0.0616, 0.05494, 0.05015, 0.04387,
        0.03745, 0.03444, 0.03289, 0.0321, 0.03164, 0.03185, 0.0332,
        0.03476,
      }, {
        9797.0, 9697.0, 9598.0, 10930.0, 4214.0, 2039.0, 704.2, 324.4, 176.1,
        106.4, 47.58, 42.42, 224.0, 152.4, 57.34, 27.22, 8.982, 3.967,
        2.078, 1.219, 0.5259, 0.2781, 0.09812, 0.05649, 0.03659,
        0.03209, 0.03036, 0.02937, 0.02795, 0.02674, 0.02536, 0.0242,
        0.02257, 0.02107, 0.02066, 0.0207, 0.02094, 0.02163, 0.02234,
        0.02368, 0.02446,
      }
    },


    // Z = 29
    {
      41, {
        0.001, 0.00104695, 0.0010961, 0.0010961, 0.0015, 0.002, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.0089789, 0.0089789, 0.01, 0.015, 0.02,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        10570.0, 9307.0, 8242.0, 9347.0, 4418.0, 2154.0, 748.8, 347.3, 189.9,
        115.6, 52.55, 38.29, 278.4, 215.9, 74.05, 33.79, 10.92, 4.862,
        2.613, 1.593, 0.763, 0.4584, 0.2217, 0.1559, 0.1119, 0.09413,
        0.08362, 0.07625, 0.06605, 0.05901, 0.05261, 0.04803, 0.04205,
        0.03599, 0.03318, 0.03177, 0.03108, 0.03074, 0.03103, 0.03247,
        0.03408,
      }, {
        10490.0, 9241.0, 8186.0, 9282.0, 4393.0, 2142.0, 743.0, 343.2, 186.6,
        112.8, 50.54, 36.52, 182.4, 148.4, 57.88, 27.88, 9.349, 4.163,
        2.192, 1.29, 0.5581, 0.2949, 0.1027, 0.05781, 0.03617, 0.03121,
        0.02933, 0.02826, 0.02681, 0.02562, 0.02428, 0.02316, 0.0216,
        0.02023, 0.01989, 0.01998, 0.02027, 0.021, 0.02174, 0.02309,
        0.02387,
      }
    },


    // Z = 30
    {
      47, {
        0.001, 0.0010098, 0.0010197, 0.0010197, 0.00103119, 0.0010428,
        0.0010428, 0.00111565, 0.0011936, 0.0011936, 0.0015, 0.002,
        0.003, 0.004, 0.005, 0.006, 0.008, 0.0096586, 0.0096586, 0.01,
        0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0,
        8.0, 10.0, 15.0, 20.0,
      }, {
        1553.0, 1518.0, 1484.0, 3804.0, 5097.0, 6518.0, 8274.0, 8452.0,
        7371.0, 8396.0, 4825.0, 2375.0, 831.1, 386.5, 211.8, 129.0,
        58.75, 35.05, 253.6, 233.1, 81.17, 37.19, 12.07, 5.384, 2.892,
        1.76, 0.8364, 0.4973, 0.2341, 0.1617, 0.1141, 0.09539, 0.0845,
        0.07695, 0.06656, 0.05941, 0.05296, 0.04834, 0.04235, 0.03634,
        0.0336, 0.03225, 0.0316, 0.03138, 0.03175, 0.03335, 0.03509,
      }, {
        1548.0, 1513.0, 1478.0, 3777.0, 5057.0, 6464.0, 8202.0, 8382.0,
        7312.0, 8328.0, 4791.0, 2359.0, 824.4, 382.0, 208.2, 126.1,
        56.6, 33.32, 159.9, 149.7, 60.99, 29.86, 10.18, 4.57, 2.419,
        1.429, 0.6203, 0.3278, 0.1128, 0.06225, 0.03764, 0.03195,
        0.02979, 0.02861, 0.02704, 0.0258, 0.02443, 0.02329, 0.02174,
        0.02041, 0.02014, 0.02028, 0.02061, 0.02143, 0.02223, 0.02367,
        0.02451,
      }
    },


    // Z = 31
    {
      47, {
        0.001, 0.00105613, 0.0011154, 0.0011154, 0.00112877, 0.0011423,
        0.0011423, 0.00121752, 0.0012977, 0.0012977, 0.0015, 0.002,
        0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.0103671, 0.0103671,
        0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0,
        8.0, 10.0, 15.0, 20.0,
      }, {
        1697.0, 1492.0, 1312.0, 3990.0, 4887.0, 5664.0, 7405.0, 7138.0,
        6358.0, 7206.0, 5087.0, 2515.0, 885.7, 413.0, 226.6, 138.2,
        63.02, 34.21, 30.99, 221.4, 85.37, 39.28, 12.81, 5.726, 3.076,
        1.868, 0.8823, 0.5197, 0.2387, 0.1619, 0.1123, 0.09325, 0.08236,
        0.07487, 0.06466, 0.05767, 0.05139, 0.04692, 0.04113, 0.03538,
        0.0328, 0.03156, 0.03099, 0.03086, 0.0313, 0.033, 0.03479,
      }, {
        1692.0, 1487.0, 1307.0, 3955.0, 4842.0, 5611.0, 7332.0, 7070.0,
        6299.0, 7139.0, 5044.0, 2496.0, 878.2, 408.2, 222.9, 135.2,
        60.82, 32.5, 29.36, 134.4, 61.35, 30.59, 10.61, 4.81, 2.56,
        1.518, 0.6613, 0.3497, 0.1193, 0.06463, 0.03782, 0.03156,
        0.0292, 0.02793, 0.02631, 0.02506, 0.02371, 0.0226, 0.0211,
        0.01986, 0.01966, 0.01985, 0.02021, 0.02108, 0.02191, 0.02339,
        0.02425,
      }
    },


    // Z = 32
    {
      47, {
        0.001, 0.00110304, 0.0012167, 0.0012167, 0.00123215, 0.0012478,
        0.0012478, 0.00132844, 0.0014143, 0.0014143, 0.0015, 0.002,
        0.003, 0.004, 0.005, 0.006, 0.008, 0.01, 0.0111031, 0.0111031,
        0.015, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0,
        8.0, 10.0, 15.0, 20.0,
      }, {
        1893.0, 1502.0, 1190.0, 4389.0, 4734.0, 4974.0, 6698.0, 6348.0,
        5554.0, 6287.0, 5475.0, 2711.0, 961.3, 449.7, 247.2, 150.9,
        68.9, 37.42, 28.11, 198.1, 91.52, 42.22, 13.85, 6.207, 3.335,
        2.023, 0.9501, 0.555, 0.2491, 0.1661, 0.1131, 0.09327, 0.08212,
        0.07452, 0.06426, 0.05727, 0.05101, 0.04657, 0.04086, 0.03524,
        0.03275, 0.03158, 0.03107, 0.03103, 0.03156, 0.0334, 0.03528,
      }, {
        1887.0, 1496.0, 1185.0, 4343.0, 4684.0, 4922.0, 6622.0, 6279.0,
        5495.0, 6221.0, 5420.0, 2688.0, 952.7, 444.5, 243.3, 147.7,
        66.6, 35.64, 26.53, 115.7, 62.56, 31.78, 11.26, 5.152, 2.759,
        1.642, 0.7184, 0.3803, 0.1288, 0.06865, 0.03891, 0.03193,
        0.0293, 0.0279, 0.02618, 0.02489, 0.02353, 0.02242, 0.02094,
        0.01977, 0.01962, 0.01987, 0.02027, 0.0212, 0.02208, 0.02364,
        0.02452,
      }
    },


    // Z = 33
    {
      46, {
        0.001, 0.00115026, 0.0013231, 0.0013231, 0.00134073, 0.0013586,
        0.0013586, 0.0015, 0.0015265, 0.0015265, 0.002, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.0118667, 0.0118667, 0.015, 0.02,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        2121.0, 1523.0, 1092.0, 4513.0, 4469.0, 4452.0, 6093.0, 5227.0,
        4997.0, 5653.0, 2931.0, 1049.0, 492.0, 270.9, 165.6, 75.73,
        41.15, 25.77, 179.2, 98.56, 45.64, 15.06, 6.76, 3.635, 2.203,
        1.03, 0.5971, 0.2622, 0.1719, 0.115, 0.09414, 0.08259, 0.07483,
        0.0644, 0.05735, 0.05106, 0.04661, 0.04093, 0.03539, 0.03296,
        0.03187, 0.03141, 0.03146, 0.03207, 0.03405, 0.03603,
      }, {
        2116.0, 1517.0, 1087.0, 4459.0, 4416.0, 4398.0, 6015.0, 5164.0,
        4938.0, 5585.0, 2902.0, 1039.0, 486.2, 266.7, 162.2, 73.3,
        39.28, 24.25, 100.8, 63.71, 33.1, 11.98, 5.537, 2.983, 1.783,
        0.7837, 0.4155, 0.1399, 0.0735, 0.04043, 0.03261, 0.02966,
        0.02813, 0.02628, 0.02494, 0.02355, 0.02243, 0.02096, 0.01984,
        0.01974, 0.02005, 0.02049, 0.02148, 0.02241, 0.02404, 0.02495,
      }
    },


    // Z = 34
    {
      46, {
        0.001, 0.00119825, 0.0014358, 0.0014358, 0.00145586, 0.0014762,
        0.0014762, 0.0015, 0.0016539, 0.0016539, 0.002, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.0126578, 0.0126578, 0.015, 0.02,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        2317.0, 1512.0, 981.4, 4347.0, 4057.0, 3907.0, 5186.0, 5336.0, 4342.0,
        4915.0, 3098.0, 1116.0, 525.2, 289.6, 177.3, 81.16, 44.14,
        23.18, 158.9, 103.3, 48.18, 15.96, 7.184, 3.864, 2.341, 1.09,
        0.6278, 0.2703, 0.1742, 0.1144, 0.09299, 0.08129, 0.0735,
        0.06314, 0.05619, 0.04999, 0.04564, 0.0401, 0.03476, 0.03247,
        0.03145, 0.03105, 0.03119, 0.03186, 0.03395, 0.03599,
      }, {
        2312.0, 1506.0, 976.0, 4287.0, 4002.0, 3855.0, 5112.0, 5260.0, 4284.0,
        4849.0, 3062.0, 1104.0, 518.7, 285.1, 173.7, 78.65, 42.21,
        21.73, 85.99, 62.7, 33.52, 12.4, 5.796, 3.143, 1.886, 0.8332,
        0.4426, 0.1483, 0.07695, 0.04113, 0.03261, 0.02941, 0.02775,
        0.02581, 0.02446, 0.02306, 0.02195, 0.02052, 0.01947, 0.01944,
        0.01978, 0.02025, 0.0213, 0.02226, 0.02394, 0.02487,
      }
    },


    // Z = 35
    {
      46, {
        0.001, 0.0015, 0.0015499, 0.0015499, 0.00157278, 0.001596, 0.001596,
        0.00168644, 0.001782, 0.001782, 0.002, 0.003, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.0134737, 0.0134737, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8,
        1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2624.0, 1002.0, 925.5, 4289.0, 3838.0, 3587.0, 5097.0, 4595.0, 3969.0,
        4495.0, 3407.0, 1231.0, 581.5, 321.3, 196.8, 90.26, 49.12,
        21.76, 147.1, 111.9, 52.66, 17.53, 7.9, 4.264, 2.582, 1.198,
        0.6861, 0.2899, 0.1838, 0.1186, 0.09563, 0.08328, 0.07515,
        0.06443, 0.05728, 0.05094, 0.0465, 0.04089, 0.03552, 0.03327,
        0.03229, 0.03194, 0.03218, 0.03293, 0.03521, 0.03738,
      }, {
        2618.0, 996.4, 920.0, 4223.0, 3780.0, 3534.0, 5014.0, 4523.0, 3910.0,
        4427.0, 3360.0, 1216.0, 574.0, 316.3, 193.0, 87.56, 47.06,
        20.33, 76.68, 63.36, 35.02, 13.28, 6.267, 3.431, 2.068, 0.9185,
        0.489, 0.1634, 0.08378, 0.0436, 0.03398, 0.03036, 0.02851,
        0.0264, 0.02496, 0.0235, 0.02237, 0.02092, 0.01991, 0.01995,
        0.02036, 0.0209, 0.02207, 0.02314, 0.02507, 0.02618,
      }
    },


    // Z = 36
    {
      46, {
        0.001, 0.0015, 0.0016749, 0.0016749, 0.00170085, 0.0017272, 0.0017272,
        0.00182152, 0.001921, 0.001921, 0.002, 0.003, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.0143256, 0.0143256, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8,
        1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2854.0, 1093.0, 836.1, 3909.0, 3409.0, 3166.0, 4566.0, 4014.0, 3482.0,
        3948.0, 3599.0, 1305.0, 618.6, 342.5, 210.1, 96.51, 52.57,
        19.71, 131.3, 116.8, 55.48, 18.54, 8.389, 4.523, 2.739, 1.267,
        0.7221, 0.2998, 0.1872, 0.1186, 0.0948, 0.08226, 0.0741, 0.0634,
        0.05631, 0.05005, 0.04569, 0.0402, 0.03501, 0.03286, 0.03196,
        0.03168, 0.03199, 0.0328, 0.03518, 0.03741,
      }, {
        2848.0, 1087.0, 830.5, 3842.0, 3352.0, 3114.0, 4484.0, 3945.0, 3425.0,
        3882.0, 3541.0, 1288.0, 610.1, 337.1, 206.0, 93.71, 50.44,
        18.36, 66.04, 61.12, 35.09, 13.65, 6.538, 3.596, 2.178, 0.9729,
        0.5192, 0.1731, 0.08787, 0.04459, 0.03414, 0.03023, 0.02825,
        0.02603, 0.02456, 0.0231, 0.02197, 0.02055, 0.01961, 0.01969,
        0.02014, 0.02071, 0.02192, 0.02302, 0.02497, 0.0261,
      }
    },


    // Z = 37
    {
      45, {
        0.001, 0.0015, 0.0018044, 0.0018044, 0.00183391, 0.0018639, 0.0018639,
        0.002, 0.0020651, 0.0020651, 0.003, 0.004, 0.005, 0.006, 0.008,
        0.01, 0.015, 0.0151997, 0.0151997, 0.02, 0.03, 0.04, 0.05, 0.06,
        0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5,
        2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        3174.0, 1219.0, 778.2, 3096.0, 2969.0, 2861.0, 3957.0, 3410.0, 3153.0,
        3606.0, 1418.0, 674.8, 374.4, 230.0, 105.8, 57.66, 19.09, 18.42,
        120.8, 59.8, 20.09, 9.112, 4.918, 2.979, 1.375, 0.7799, 0.3187,
        0.196, 0.1219, 0.0967, 0.0836, 0.0751, 0.06412, 0.05689,
        0.05053, 0.04613, 0.04061, 0.03545, 0.03335, 0.0325, 0.03227,
        0.03267, 0.03357, 0.0361, 0.03845,
      }, {
        3168.0, 1213.0, 772.6, 3040.0, 2915.0, 2810.0, 3880.0, 3347.0, 3096.0,
        3540.0, 1397.0, 664.8, 368.2, 225.5, 102.8, 55.41, 17.75, 17.09,
        58.64, 35.81, 14.36, 6.963, 3.858, 2.349, 1.056, 0.5649, 0.1881,
        0.09465, 0.0469, 0.03531, 0.03098, 0.02878, 0.02638, 0.02484,
        0.02332, 0.02217, 0.02074, 0.01983, 0.01996, 0.02045, 0.02105,
        0.02231, 0.02343, 0.02538, 0.02645,
      }
    },


    // Z = 38
    {
      45, {
        0.001, 0.0015, 0.0019396, 0.0019396, 0.002, 0.0020068, 0.0020068,
        0.00210895, 0.0022163, 0.0022163, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.01, 0.015, 0.0161046, 0.0161046, 0.02, 0.03, 0.04,
        0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        3494.0, 1347.0, 720.7, 2864.0, 2589.0, 2571.0, 3577.0, 3201.0, 2842.0,
        3241.0, 1525.0, 729.7, 405.8, 249.6, 115.0, 62.74, 20.79, 17.14,
        110.8, 63.86, 21.57, 9.818, 5.306, 3.214, 1.481, 0.8365, 0.3369,
        0.2042, 0.1247, 0.09811, 0.08443, 0.0757, 0.06447, 0.05714,
        0.05072, 0.0463, 0.04079, 0.03569, 0.03365, 0.03286, 0.03268,
        0.03317, 0.03414, 0.03683, 0.03927,
      }, {
        3487.0, 1341.0, 715.2, 2807.0, 2539.0, 2521.0, 3501.0, 3134.0, 2786.0,
        3176.0, 1500.0, 718.1, 398.8, 244.6, 111.8, 60.37, 19.38, 15.85,
        52.0, 36.02, 14.93, 7.346, 4.103, 2.511, 1.135, 0.6093, 0.2028,
        0.1013, 0.04906, 0.03634, 0.03156, 0.02917, 0.0266, 0.02498,
        0.02342, 0.02225, 0.02082, 0.01995, 0.02013, 0.02066, 0.0213,
        0.0226, 0.02377, 0.02578, 0.02684,
      }
    },


    // Z = 39
    {
      46, {
        0.001, 0.0015, 0.002, 0.00208, 0.00208, 0.00211741, 0.0021555,
        0.0021555, 0.0022614, 0.0023725, 0.0023725, 0.003, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.0170384, 0.0170384, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8,
        1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        3864.0, 1493.0, 742.2, 673.8, 2627.0, 2466.0, 2342.0, 3264.0, 2916.0,
        2597.0, 2962.0, 1654.0, 793.6, 442.4, 272.5, 125.8, 68.71,
        22.79, 16.12, 102.9, 68.55, 23.3, 10.65, 5.764, 3.493, 1.607,
        0.9047, 0.3595, 0.2149, 0.1289, 0.1006, 0.08613, 0.07703,
        0.06546, 0.05795, 0.05141, 0.04692, 0.04137, 0.03628, 0.03428,
        0.03355, 0.03341, 0.03399, 0.03504, 0.0379, 0.04048,
      }, {
        3857.0, 1487.0, 736.5, 668.2, 2569.0, 2412.0, 2292.0, 3187.0, 2849.0,
        2540.0, 2897.0, 1623.0, 779.8, 434.4, 267.0, 122.4, 66.19, 21.3,
        14.86, 46.72, 36.2, 15.58, 7.79, 4.388, 2.699, 1.228, 0.6616,
        0.2203, 0.1093, 0.05186, 0.03779, 0.03249, 0.02986, 0.02708,
        0.02537, 0.02375, 0.02254, 0.02109, 0.02026, 0.02048, 0.02107,
        0.02174, 0.02311, 0.02432, 0.02638, 0.02746,
      }
    },


    // Z = 40
    {
      46, {
        0.001, 0.0015, 0.002, 0.0022223, 0.0022223, 0.00226411, 0.0023067,
        0.0023067, 0.00241654, 0.0025316, 0.0025316, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.015, 0.0179976, 0.0179976, 0.02,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        4210.0, 1631.0, 811.5, 625.8, 2392.0, 2239.0, 2120.0, 2953.0, 2641.0,
        2359.0, 2691.0, 1772.0, 850.7, 475.5, 293.5, 135.6, 74.17,
        24.63, 15.01, 94.7, 72.37, 24.85, 11.39, 6.173, 3.744, 1.721,
        0.9658, 0.379, 0.2237, 0.1318, 0.1018, 0.08693, 0.07756,
        0.06571, 0.0581, 0.0515, 0.047, 0.04146, 0.03644, 0.03451,
        0.03384, 0.03374, 0.03441, 0.03554, 0.03855, 0.04122,
      }, {
        4203.0, 1625.0, 805.7, 620.1, 2335.0, 2186.0, 2071.0, 2878.0, 2575.0,
        2303.0, 2627.0, 1734.0, 834.4, 466.4, 287.3, 132.0, 71.5, 23.08,
        13.8, 41.64, 35.55, 16.0, 8.129, 4.621, 2.859, 1.31, 0.708,
        0.2361, 0.1164, 0.0542, 0.03885, 0.03311, 0.03025, 0.02726,
        0.02547, 0.0238, 0.02257, 0.02112, 0.02033, 0.02061, 0.02123,
        0.02193, 0.02335, 0.0246, 0.0267, 0.02778,
      }
    },


    // Z = 41
    {
      46, {
        0.001, 0.0015, 0.002, 0.0023705, 0.0023705, 0.00241714, 0.0024647,
        0.0024647, 0.00257857, 0.0026977, 0.0026977, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.015, 0.0189856, 0.0189856, 0.02,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        4600.0, 1786.0, 889.3, 584.4, 2181.0, 2045.0, 1935.0, 2694.0, 2412.0,
        2161.0, 2470.0, 1906.0, 916.4, 513.4, 317.2, 146.9, 80.38,
        26.72, 14.09, 87.84, 77.12, 26.66, 12.23, 6.644, 4.032, 1.852,
        1.037, 0.4023, 0.2344, 0.1357, 0.104, 0.08831, 0.07858, 0.06642,
        0.05866, 0.05196, 0.04741, 0.04185, 0.03686, 0.03498, 0.03436,
        0.03432, 0.03508, 0.03628, 0.03944, 0.04224,
      }, {
        4592.0, 1780.0, 883.2, 578.6, 2125.0, 1992.0, 1886.0, 2619.0, 2347.0,
        2104.0, 2405.0, 1859.0, 897.0, 502.7, 310.3, 142.9, 77.54,
        25.08, 12.91, 37.46, 34.93, 16.48, 8.511, 4.884, 3.04, 1.403,
        0.7608, 0.2542, 0.1247, 0.05705, 0.04026, 0.03396, 0.03085,
        0.02764, 0.02575, 0.02402, 0.02276, 0.0213, 0.02054, 0.02086,
        0.02153, 0.02227, 0.02375, 0.02503, 0.02718, 0.02828,
      }
    },


    // Z = 42
    {
      46, {
        0.001, 0.0015, 0.002, 0.0025202, 0.0025202, 0.00257212, 0.0026251,
        0.0026251, 0.00274267, 0.0028655, 0.0028655, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.015, 0.0199995, 0.0199995, 0.02,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        4942.0, 1925.0, 959.3, 541.5, 1979.0, 1854.0, 1750.0, 2433.0, 2183.0,
        1961.0, 2243.0, 2011.0, 970.3, 545.0, 337.3, 156.5, 85.76,
        28.54, 13.08, 80.55, 80.54, 28.1, 12.94, 7.037, 4.274, 1.962,
        1.096, 0.4208, 0.2423, 0.1379, 0.1047, 0.08848, 0.07851,
        0.06619, 0.05837, 0.05167, 0.04713, 0.04163, 0.03675, 0.03496,
        0.03439, 0.0344, 0.03523, 0.0365, 0.03978, 0.04264,
      }, {
        4935.0, 1918.0, 953.1, 535.8, 1924.0, 1802.0, 1703.0, 2360.0, 2119.0,
        1906.0, 2179.0, 1956.0, 947.5, 532.8, 329.5, 152.2, 82.75,
        26.84, 11.93, 32.93, 33.36, 16.64, 8.757, 5.074, 3.178, 1.477,
        0.8042, 0.2693, 0.1316, 0.05919, 0.04117, 0.03437, 0.03104,
        0.02764, 0.02567, 0.0239, 0.02263, 0.02118, 0.02046, 0.02084,
        0.02153, 0.02231, 0.02382, 0.02513, 0.02731, 0.0284,
      }
    },


    // Z = 43
    {
      45, {
        0.001, 0.0015, 0.002, 0.0026769, 0.0026769, 0.00273443, 0.0027932,
        0.0027932, 0.003, 0.0030425, 0.0030425, 0.004, 0.005, 0.006,
        0.008, 0.01, 0.015, 0.02, 0.021044, 0.021044, 0.03, 0.04, 0.05,
        0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25,
        1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        5356.0, 2092.0, 1044.0, 507.2, 1812.0, 1699.0, 1602.0, 2223.0, 1862.0,
        1800.0, 2059.0, 1036.0, 583.6, 361.9, 168.3, 92.31, 30.76, 14.1,
        12.29, 74.81, 29.93, 13.81, 7.521, 4.571, 2.099, 1.169, 0.4449,
        0.2534, 0.1418, 0.1066, 0.08968, 0.07935, 0.06674, 0.05876,
        0.05197, 0.0474, 0.04189, 0.03705, 0.03532, 0.03482, 0.03487,
        0.03578, 0.03712, 0.04055, 0.04353,
      }, {
        5348.0, 2085.0, 1037.0, 501.4, 1758.0, 1648.0, 1555.0, 2151.0, 1805.0,
        1745.0, 1995.0, 1009.0, 569.4, 353.0, 163.5, 89.08, 28.98, 12.9,
        11.17, 30.14, 16.93, 9.075, 5.312, 3.348, 1.568, 0.8572, 0.288,
        0.1403, 0.06214, 0.04257, 0.03521, 0.03159, 0.02797, 0.02589,
        0.02406, 0.02276, 0.02129, 0.02061, 0.02103, 0.02178, 0.02258,
        0.02415, 0.02549, 0.02774, 0.02884,
      }
    },


    // Z = 44
    {
      45, {
        0.001, 0.0015, 0.002, 0.0028379, 0.0028379, 0.00290168, 0.0029669,
        0.0029669, 0.003, 0.003224, 0.003224, 0.004, 0.005, 0.006,
        0.008, 0.01, 0.015, 0.02, 0.0221172, 0.0221172, 0.03, 0.04,
        0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        5718.0, 2240.0, 1120.0, 470.4, 1644.0, 1542.0, 1452.0, 2317.0, 1963.0,
        1638.0, 1874.0, 1095.0, 616.5, 383.2, 178.5, 98.0, 32.7, 14.99,
        11.43, 68.76, 31.39, 14.52, 7.926, 4.822, 2.215, 1.232, 0.4647,
        0.2618, 0.144, 0.1074, 0.08992, 0.07933, 0.06647, 0.05846,
        0.05166, 0.0471, 0.04164, 0.03692, 0.03527, 0.03482, 0.03491,
        0.03591, 0.0373, 0.04084, 0.04387,
      }, {
        5709.0, 2233.0, 1113.0, 464.6, 1591.0, 1492.0, 1407.0, 2233.0, 1894.0,
        1584.0, 1811.0, 1063.0, 600.0, 373.1, 173.3, 94.56, 30.85,
        13.76, 10.35, 27.0, 16.89, 9.251, 5.477, 3.476, 1.641, 0.9012,
        0.3037, 0.1475, 0.06445, 0.04355, 0.03567, 0.03182, 0.02796,
        0.02581, 0.02393, 0.02262, 0.02115, 0.02052, 0.02098, 0.02176,
        0.02259, 0.02419, 0.02556, 0.02783, 0.02893,
      }
    },


    // Z = 45
    {
      46, {
        0.001, 0.0015, 0.002, 0.003, 0.0030038, 0.0030038, 0.00307413,
        0.0031461, 0.0031461, 0.00327631, 0.0034119, 0.0034119, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.015, 0.02, 0.0232199, 0.0232199,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        6169.0, 2426.0, 1214.0, 444.1, 442.7, 1513.0, 1422.0, 1337.0, 1847.0,
        1671.0, 1512.0, 1731.0, 1170.0, 658.9, 410.1, 191.5, 105.3,
        35.18, 16.13, 10.79, 64.14, 33.3, 15.44, 8.448, 5.147, 2.365,
        1.314, 0.4916, 0.2742, 0.1485, 0.1097, 0.09134, 0.08035,
        0.06711, 0.05894, 0.05204, 0.04744, 0.04197, 0.03728, 0.03568,
        0.03529, 0.03542, 0.0365, 0.03797, 0.04166, 0.04481,
      }, {
        6160.0, 2419.0, 1207.0, 438.2, 436.8, 1461.0, 1372.0, 1292.0, 1777.0,
        1608.0, 1458.0, 1669.0, 1132.0, 639.6, 398.5, 185.7, 101.5,
        33.22, 14.84, 9.736, 24.58, 16.99, 9.52, 5.706, 3.649, 1.737,
        0.9581, 0.3241, 0.1571, 0.06776, 0.04517, 0.03663, 0.03247,
        0.02834, 0.02608, 0.02413, 0.02278, 0.02131, 0.02071, 0.02122,
        0.02204, 0.0229, 0.02456, 0.02597, 0.02831, 0.02943,
      }
    },


    // Z = 46
    {
      46, {
        0.001, 0.0015, 0.002, 0.003, 0.0031733, 0.0031733, 0.00325085,
        0.0033303, 0.0033303, 0.00346459, 0.0036043, 0.0036043, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.015, 0.02, 0.0243503, 0.0243503,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        6538.0, 2579.0, 1292.0, 473.0, 410.6, 1355.0, 1287.0, 1215.0, 1664.0,
        1518.0, 1379.0, 1582.0, 1227.0, 691.2, 430.8, 201.7, 111.0,
        37.15, 17.04, 10.03, 58.98, 34.65, 16.14, 8.85, 5.399, 2.481,
        1.377, 0.5115, 0.2827, 0.1506, 0.1103, 0.09134, 0.0801, 0.06669,
        0.05849, 0.05159, 0.04702, 0.04162, 0.03704, 0.03552, 0.03518,
        0.03537, 0.03651, 0.03802, 0.04181, 0.04501,
      }, {
        6529.0, 2571.0, 1285.0, 466.8, 404.7, 1305.0, 1238.0, 1171.0, 1597.0,
        1456.0, 1326.0, 1520.0, 1182.0, 668.8, 417.5, 195.3, 107.0,
        35.11, 15.71, 9.016, 22.08, 16.69, 9.611, 5.833, 3.759, 1.804,
        1.0, 0.3397, 0.1644, 0.07007, 0.04611, 0.03704, 0.03262,
        0.02828, 0.02594, 0.02395, 0.02259, 0.02112, 0.02056, 0.02111,
        0.02196, 0.02284, 0.02453, 0.02595, 0.02832, 0.02944,
      }
    },


    // Z = 47
    {
      46, {
        0.001, 0.0015, 0.002, 0.003, 0.0033511, 0.0033511, 0.00343632,
        0.0035237, 0.0035237, 0.00366203, 0.0038058, 0.0038058, 0.004,
        0.005, 0.006, 0.008, 0.01, 0.015, 0.02, 0.025514, 0.025514,
        0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        7039.0, 2790.0, 1401.0, 513.6, 388.7, 1274.0, 1200.0, 1126.0, 1547.0,
        1409.0, 1282.0, 1468.0, 1305.0, 738.5, 461.0, 216.4, 119.3,
        39.98, 18.36, 9.527, 55.39, 36.68, 17.2, 9.444, 5.766, 2.651,
        1.47, 0.5426, 0.2972, 0.156, 0.1131, 0.09321, 0.08153, 0.06766,
        0.05921, 0.05217, 0.04754, 0.04209, 0.03754, 0.03606, 0.03577,
        0.03601, 0.03723, 0.03882, 0.04276, 0.04609,
      }, {
        7029.0, 2782.0, 1393.0, 507.3, 382.7, 1223.0, 1151.0, 1082.0, 1479.0,
        1348.0, 1229.0, 1406.0, 1252.0, 712.2, 445.7, 209.2, 114.9,
        37.8, 16.95, 8.534, 20.31, 16.6, 9.869, 6.066, 3.938, 1.907,
        1.062, 0.3622, 0.1751, 0.07392, 0.04803, 0.03822, 0.03347,
        0.02881, 0.02632, 0.02425, 0.02284, 0.02135, 0.02082, 0.02142,
        0.02232, 0.02324, 0.02499, 0.02647, 0.02889, 0.03005,
      }
    },


    // Z = 48
    {
      45, {
        0.001, 0.0015, 0.002, 0.003, 0.0035375, 0.0035375, 0.00363101,
        0.003727, 0.003727, 0.004, 0.004018, 0.004018, 0.005, 0.006,
        0.008, 0.01, 0.015, 0.02, 0.0267112, 0.0267112, 0.03, 0.04,
        0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        7350.0, 2931.0, 1473.0, 541.4, 357.5, 1152.0, 1083.0, 1013.0, 1389.0,
        1170.0, 1157.0, 1324.0, 768.5, 479.3, 225.4, 124.4, 41.78, 19.2,
        8.809, 50.65, 37.65, 17.78, 9.779, 5.975, 2.751, 1.524, 0.5593,
        0.3038, 0.1571, 0.1129, 0.0925, 0.08064, 0.0667, 0.05826,
        0.05129, 0.04673, 0.04139, 0.03698, 0.03559, 0.03536, 0.03563,
        0.03691, 0.03853, 0.04253, 0.04587,
      }, {
        7340.0, 2922.0, 1466.0, 535.0, 351.7, 1103.0, 1036.0, 971.5, 1324.0,
        1118.0, 1106.0, 1264.0, 738.3, 461.9, 217.4, 119.7, 39.52,
        17.76, 7.863, 18.21, 15.94, 9.812, 6.115, 4.001, 1.957, 1.096,
        0.3753, 0.1813, 0.0758, 0.04868, 0.03838, 0.03339, 0.02854,
        0.02597, 0.02387, 0.02247, 0.02099, 0.02051, 0.02114, 0.02206,
        0.023, 0.02477, 0.02625, 0.02869, 0.02985,
      }
    },


    // Z = 49
    {
      45, {
        0.001, 0.0015, 0.002, 0.003, 0.0037301, 0.0037301, 0.00383264,
        0.003938, 0.003938, 0.004, 0.0042375, 0.0042375, 0.005, 0.006,
        0.008, 0.01, 0.015, 0.02, 0.0279399, 0.0279399, 0.03, 0.04,
        0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        7809.0, 3131.0, 1578.0, 580.8, 335.6, 1046.0, 993.0, 931.3, 1261.0,
        1231.0, 1066.0, 1223.0, 813.4, 507.2, 239.1, 132.1, 44.45,
        20.44, 8.316, 47.33, 39.49, 18.73, 10.3, 6.306, 2.907, 1.609,
        0.5876, 0.3167, 0.1614, 0.1149, 0.09371, 0.08138, 0.06707,
        0.05849, 0.05144, 0.04684, 0.04151, 0.03715, 0.03582, 0.03564,
        0.03596, 0.0373, 0.03898, 0.04311, 0.04654,
      }, {
        7797.0, 3122.0, 1570.0, 574.3, 329.8, 998.9, 946.5, 890.5, 1199.0,
        1170.0, 1016.0, 1164.0, 778.0, 487.1, 230.1, 126.9, 42.06,
        18.93, 7.395, 16.7, 15.53, 9.911, 6.262, 4.135, 2.043, 1.15,
        0.396, 0.1913, 0.07923, 0.05031, 0.03932, 0.03398, 0.02883,
        0.02615, 0.02398, 0.02254, 0.02105, 0.0206, 0.02127, 0.02223,
        0.02321, 0.02502, 0.02653, 0.02903, 0.0302,
      }
    },


    // Z = 50
    {
      45, {
        0.001, 0.0015, 0.002, 0.003, 0.0039288, 0.0039288, 0.004, 0.0041561,
        0.0041561, 0.00430764, 0.0044647, 0.0044647, 0.005, 0.006,
        0.008, 0.01, 0.015, 0.02, 0.0292001, 0.0292001, 0.03, 0.04,
        0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        8157.0, 3296.0, 1665.0, 614.3, 311.4, 928.5, 939.3, 846.9, 1145.0,
        1060.0, 971.2, 1117.0, 847.1, 529.4, 250.0, 138.4, 46.64, 21.46,
        7.76, 43.6, 41.21, 19.42, 10.7, 6.564, 3.029, 1.676, 0.6091,
        0.326, 0.1639, 0.1156, 0.09374, 0.08113, 0.06662, 0.058,
        0.05095, 0.04638, 0.04112, 0.03686, 0.03561, 0.03548, 0.03583,
        0.03724, 0.03895, 0.04315, 0.04662,
      }, {
        8144.0, 3287.0, 1657.0, 607.7, 305.8, 884.5, 894.5, 807.4, 1084.0,
        1002.0, 922.1, 1059.0, 806.4, 506.5, 240.0, 132.7, 44.13, 19.9,
        6.876, 15.14, 14.9, 9.835, 6.314, 4.211, 2.101, 1.189, 0.4119,
        0.199, 0.08179, 0.05138, 0.0398, 0.03417, 0.02878, 0.02601,
        0.02379, 0.02233, 0.02085, 0.02044, 0.02115, 0.02213, 0.02312,
        0.02496, 0.02649, 0.02901, 0.03018,
      }
    },


    // Z = 51
    {
      46, {
        0.001, 0.0015, 0.002, 0.003, 0.004, 0.0041322, 0.0041322, 0.00425449,
        0.0043804, 0.0043804, 0.00453657, 0.0046983, 0.0046983, 0.005,
        0.006, 0.008, 0.01, 0.015, 0.02, 0.03, 0.0304912, 0.0304912,
        0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8,
        1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        8582.0, 3491.0, 1767.0, 653.6, 316.9, 291.8, 869.1, 830.8, 777.6,
        1050.0, 974.3, 893.9, 1029.0, 884.6, 556.9, 263.1, 145.9, 49.23,
        22.68, 7.631, 7.307, 40.73, 20.27, 11.2, 6.879, 3.176, 1.758,
        0.6361, 0.3381, 0.1677, 0.1172, 0.09453, 0.08153, 0.0667,
        0.05797, 0.05086, 0.04628, 0.04105, 0.03686, 0.03567, 0.03559,
        0.03598, 0.03745, 0.03921, 0.04351, 0.04704,
      }, {
        8568.0, 3481.0, 1759.0, 646.9, 311.3, 286.3, 825.2, 786.5, 739.2,
        990.6, 917.8, 845.7, 972.5, 837.7, 530.5, 251.8, 139.6, 46.57,
        21.05, 6.755, 6.452, 13.91, 9.789, 6.4, 4.311, 2.173, 1.237,
        0.4312, 0.2084, 0.08504, 0.05288, 0.04061, 0.03465, 0.02896,
        0.02608, 0.02378, 0.0223, 0.02081, 0.02043, 0.02118, 0.02219,
        0.02321, 0.02509, 0.02664, 0.0292, 0.03038,
      }
    },


    // Z = 52
    {
      49, {
        0.001, 0.001003, 0.001006, 0.001006, 0.0015, 0.002, 0.003, 0.004,
        0.0043414, 0.0043414, 0.00447465, 0.004612, 0.004612, 0.0047728,
        0.0049392, 0.0049392, 0.005, 0.006, 0.008, 0.01, 0.015, 0.02,
        0.03, 0.0318138, 0.0318138, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15,
        0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0,
        5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        8434.0, 8380.0, 8326.0, 8684.0, 3608.0, 1832.0, 679.2, 329.7, 267.8,
        788.2, 750.4, 699.5, 944.5, 878.2, 806.2, 929.2, 901.4, 572.1,
        270.2, 150.1, 50.78, 23.41, 7.878, 6.738, 37.19, 20.64, 11.45,
        7.041, 3.255, 1.801, 0.6492, 0.3429, 0.1679, 0.1163, 0.09328,
        0.08022, 0.06538, 0.05669, 0.04967, 0.04518, 0.04009, 0.03606,
        0.03494, 0.03492, 0.03534, 0.03683, 0.0386, 0.0429, 0.04642,
      }, {
        8419.0, 8365.0, 8311.0, 8668.0, 3598.0, 1824.0, 672.6, 324.0, 262.5,
        746.0, 707.8, 662.9, 888.0, 823.7, 760.0, 874.4, 848.7, 542.3,
        257.8, 143.3, 48.02, 21.75, 6.997, 5.929, 12.51, 9.48, 6.33,
        4.306, 2.195, 1.256, 0.4409, 0.2132, 0.08646, 0.05326, 0.04055,
        0.0344, 0.02854, 0.02559, 0.02327, 0.02179, 0.02033, 0.01998,
        0.02075, 0.02177, 0.02279, 0.02466, 0.0262, 0.02873, 0.02989,
      }
    },


    // Z = 53
    {
      48, {
        0.001, 0.00103542, 0.0010721, 0.0010721, 0.0015, 0.002, 0.003, 0.004,
        0.0045571, 0.0045571, 0.00470229, 0.0048521, 0.0048521, 0.005,
        0.0051881, 0.0051881, 0.006, 0.008, 0.01, 0.015, 0.02, 0.03,
        0.0331694, 0.0331694, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2,
        0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0,
        6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        9096.0, 8465.0, 7863.0, 8198.0, 3919.0, 1997.0, 742.0, 360.7, 259.2,
        755.0, 712.3, 663.6, 894.3, 843.0, 766.5, 883.7, 617.3, 292.2,
        162.6, 55.12, 25.43, 8.561, 6.553, 35.82, 22.1, 12.32, 7.579,
        3.51, 1.942, 0.6978, 0.3663, 0.1771, 0.1217, 0.09701, 0.08313,
        0.06749, 0.05841, 0.05111, 0.04647, 0.04124, 0.03716, 0.03607,
        0.03608, 0.03655, 0.03815, 0.04002, 0.04455, 0.04823,
      }, {
        9078.0, 8448.0, 7847.0, 8181.0, 3908.0, 1988.0, 735.1, 354.8, 253.7,
        712.1, 672.4, 627.0, 837.5, 790.3, 719.8, 828.3, 582.2, 277.7,
        154.8, 52.08, 23.63, 7.622, 5.744, 11.88, 9.616, 6.573, 4.518,
        2.331, 1.342, 0.4742, 0.2295, 0.09257, 0.0565, 0.04267, 0.03598,
        0.02962, 0.02646, 0.02399, 0.02243, 0.02092, 0.02059, 0.02142,
        0.0225, 0.02357, 0.02553, 0.02714, 0.0298, 0.03101,
      }
    },


    // Z = 54
    {
      48, {
        0.001, 0.00107191, 0.001149, 0.001149, 0.0015, 0.002, 0.003, 0.004,
        0.0047822, 0.0047822, 0.005, 0.0051037, 0.0051037, 0.00527536,
        0.0054528, 0.0054528, 0.006, 0.008, 0.01, 0.015, 0.02, 0.03,
        0.0345614, 0.0345614, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2,
        0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0,
        6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        9413.0, 8151.0, 7035.0, 7338.0, 4085.0, 2088.0, 778.0, 378.7, 240.8,
        694.1, 639.2, 604.4, 818.1, 757.9, 699.1, 806.4, 637.6, 303.2,
        169.0, 57.43, 26.52, 8.93, 6.129, 33.16, 22.7, 12.72, 7.825,
        3.633, 2.011, 0.7202, 0.376, 0.1797, 0.1223, 0.09699, 0.08281,
        0.06696, 0.05785, 0.05054, 0.04594, 0.04078, 0.03681, 0.03577,
        0.03583, 0.03634, 0.03797, 0.03987, 0.04445, 0.04815,
      }, {
        9393.0, 8132.0, 7019.0, 7321.0, 4073.0, 2078.0, 770.9, 372.8, 235.5,
        652.4, 601.5, 569.3, 763.0, 707.9, 653.9, 752.7, 597.9, 287.1,
        160.5, 54.2, 24.65, 7.969, 5.352, 10.86, 9.323, 6.54, 4.541,
        2.374, 1.376, 0.4894, 0.2373, 0.09522, 0.05761, 0.04317,
        0.03617, 0.02956, 0.0263, 0.02378, 0.02221, 0.0207, 0.02042,
        0.02128, 0.0224, 0.0235, 0.02553, 0.0272, 0.03002, 0.03139,
      }
    },


    // Z = 55
    {
      52, {
        0.001, 0.00103199, 0.001065, 0.001065, 0.00113851, 0.0012171,
        0.0012171, 0.0015, 0.002, 0.003, 0.004, 0.005, 0.0050119,
        0.0050119, 0.00518274, 0.0053594, 0.0053594, 0.00553401,
        0.0057143, 0.0057143, 0.006, 0.008, 0.01, 0.015, 0.02, 0.03,
        0.0359846, 0.0359846, 0.04, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2,
        0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0,
        6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        9365.0, 8775.0, 8214.0, 8685.0, 7576.0, 6584.0, 6888.0, 4335.0,
        2226.0, 831.9, 405.5, 230.3, 229.0, 667.4, 618.4, 564.5, 769.2,
        714.6, 655.6, 754.7, 671.1, 321.4, 179.3, 61.04, 28.22, 9.507,
        5.863, 31.43, 23.81, 13.4, 8.248, 3.836, 2.124, 0.7589, 0.3941,
        0.1863, 0.1257, 0.09912, 0.08431, 0.06789, 0.05854, 0.05108,
        0.04641, 0.04121, 0.03725, 0.03625, 0.03635, 0.03689, 0.0386,
        0.04057, 0.04529, 0.0491,
      }, {
        9343.0, 8754.0, 8194.0, 8663.0, 7557.0, 6567.0, 6870.0, 4322.0,
        2216.0, 824.6, 399.4, 225.1, 223.7, 624.8, 576.3, 529.9, 714.2,
        661.0, 610.8, 701.5, 625.3, 302.9, 169.7, 57.53, 26.23, 8.502,
        5.102, 10.17, 9.185, 6.618, 4.653, 2.464, 1.438, 0.5154, 0.2502,
        0.1, 0.06003, 0.04463, 0.03717, 0.03014, 0.02671, 0.02408,
        0.02246, 0.02091, 0.02064, 0.02153, 0.02268, 0.0238, 0.02583,
        0.0275, 0.03026, 0.03152,
      }
    },


    // Z = 56
    {
      55, {
        0.001, 0.00103063, 0.0010622, 0.0010622, 0.00109882, 0.0011367,
        0.0011367, 0.00121224, 0.0012928, 0.0012928, 0.0015, 0.002,
        0.003, 0.004, 0.005, 0.005247, 0.005247, 0.00543204, 0.0056236,
        0.0056236, 0.00580333, 0.0059888, 0.0059888, 0.006, 0.008, 0.01,
        0.015, 0.02, 0.03, 0.0374406, 0.0374406, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        8543.0, 7990.0, 7465.0, 8547.0, 7957.0, 7404.0, 7837.0, 6861.0,
        5985.0, 6255.0, 4499.0, 2319.0, 869.6, 424.6, 241.4, 213.5,
        609.8, 563.4, 516.9, 701.6, 650.7, 601.3, 693.0, 689.8, 333.4,
        186.0, 63.47, 29.38, 9.904, 5.498, 29.19, 24.57, 13.79, 8.511,
        3.963, 2.196, 0.7828, 0.4046, 0.1891, 0.1265, 0.09923, 0.0841,
        0.06744, 0.05803, 0.05058, 0.04592, 0.04078, 0.03692, 0.03598,
        0.03612, 0.03669, 0.03844, 0.04042, 0.04518, 0.04902,
      }, {
        8521.0, 7969.0, 7445.0, 8524.0, 7935.0, 7384.0, 7815.0, 6841.0,
        5968.0, 6237.0, 4485.0, 2308.0, 862.2, 418.4, 236.1, 208.4,
        568.9, 526.3, 483.5, 648.6, 602.5, 557.7, 641.2, 638.3, 312.7,
        175.4, 59.72, 27.3, 8.875, 4.768, 9.346, 8.842, 6.534, 4.66,
        2.501, 1.47, 0.5309, 0.2583, 0.1028, 0.06125, 0.04521, 0.03743,
        0.03011, 0.02657, 0.02388, 0.02224, 0.02069, 0.02044, 0.02135,
        0.0225, 0.02363, 0.02566, 0.02731, 0.03003, 0.03126,
      }
    },


    // Z = 57
    {
      54, {
        0.001, 0.00105991, 0.0011234, 0.0011234, 0.0011632, 0.0012044,
        0.0012044, 0.00128045, 0.0013613, 0.0013613, 0.0015, 0.002,
        0.003, 0.004, 0.005, 0.0054827, 0.0054827, 0.00568299,
        0.0058906, 0.0058906, 0.006, 0.0062663, 0.0062663, 0.008, 0.01,
        0.015, 0.02, 0.03, 0.0389246, 0.0389246, 0.04, 0.05, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        9087.0, 7988.0, 7000.0, 8025.0, 7447.0, 6911.0, 7321.0, 6440.0,
        5652.0, 5905.0, 4772.0, 2464.0, 926.7, 453.1, 257.8, 203.9,
        576.4, 530.2, 484.6, 654.5, 631.9, 565.4, 651.8, 352.9, 196.7,
        67.31, 31.19, 10.52, 5.271, 27.72, 25.79, 14.47, 8.962, 4.177,
        2.315, 0.8239, 0.4239, 0.1961, 0.1301, 0.1015, 0.0857, 0.06843,
        0.05876, 0.0511, 0.0464, 0.04122, 0.03737, 0.03646, 0.03664,
        0.03726, 0.03907, 0.04113, 0.04603, 0.04996,
      }, {
        9061.0, 7965.0, 6979.0, 8001.0, 7425.0, 6890.0, 7298.0, 6419.0,
        5634.0, 5886.0, 4755.0, 2453.0, 919.0, 446.8, 252.4, 198.8,
        535.6, 493.5, 451.7, 602.3, 581.9, 522.1, 600.3, 329.0, 184.7,
        63.21, 28.97, 9.444, 4.554, 8.785, 8.608, 6.56, 4.756, 2.587,
        1.532, 0.5576, 0.2718, 0.1079, 0.06382, 0.04677, 0.03849,
        0.03073, 0.027, 0.02418, 0.02249, 0.0209, 0.02068, 0.02161,
        0.0228, 0.02395, 0.02602, 0.02769, 0.03043, 0.03163,
      }
    },


    // Z = 58
    {
      54, {
        0.001, 0.00108876, 0.0011854, 0.0011854, 0.00122832, 0.0012728,
        0.0012728, 0.00135222, 0.0014366, 0.0014366, 0.0015, 0.002,
        0.003, 0.004, 0.005, 0.0057234, 0.0057234, 0.006, 0.0061642,
        0.0061642, 0.00635359, 0.0065488, 0.0065488, 0.008, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.040443, 0.040443, 0.05, 0.06, 0.08, 0.1,
        0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0,
        4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        9711.0, 7991.0, 6563.0, 7545.0, 6984.0, 6465.0, 6853.0, 6031.0,
        5293.0, 5532.0, 5033.0, 2607.0, 985.7, 481.1, 274.0, 194.3,
        545.4, 490.8, 455.4, 618.8, 575.6, 533.2, 614.9, 373.2, 208.2,
        71.43, 33.12, 11.19, 5.215, 5.066, 26.35, 15.2, 9.447, 4.409,
        2.445, 0.8687, 0.4452, 0.2039, 0.1342, 0.1041, 0.08757, 0.06962,
        0.05961, 0.05181, 0.04701, 0.04177, 0.03792, 0.03705, 0.03727,
        0.03792, 0.03981, 0.04194, 0.04699, 0.05103,
      }, {
        9679.0, 7964.0, 6541.0, 7519.0, 6960.0, 6442.0, 6829.0, 6010.0,
        5273.0, 5512.0, 5015.0, 2595.0, 977.6, 474.5, 268.4, 189.2,
        504.6, 455.0, 422.8, 566.5, 527.8, 489.9, 563.3, 345.8, 194.5,
        66.9, 30.73, 10.05, 4.496, 4.359, 8.272, 6.569, 4.847, 2.677,
        1.597, 0.5862, 0.2865, 0.1134, 0.06664, 0.04851, 0.0397,
        0.03145, 0.0275, 0.02456, 0.02281, 0.02118, 0.02096, 0.02194,
        0.02316, 0.02434, 0.02645, 0.02816, 0.03094, 0.03216,
      }
    },


    // Z = 59
    {
      53, {
        0.001, 0.00111454, 0.0012422, 0.0012422, 0.00128892, 0.0013374,
        0.0013374, 0.0015, 0.001511, 0.001511, 0.002, 0.003, 0.004,
        0.005, 0.0059643, 0.0059643, 0.006, 0.0064404, 0.0064404,
        0.00663467, 0.0068348, 0.0068348, 0.008, 0.01, 0.015, 0.02,
        0.03, 0.04, 0.0419906, 0.0419906, 0.05, 0.06, 0.08, 0.1, 0.15,
        0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0,
        5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        10580.0, 8122.0, 6278.0, 7224.0, 6661.0, 6142.0, 6513.0, 5090.0,
        5006.0, 5235.0, 2768.0, 1047.0, 513.1, 292.4, 186.8, 521.4,
        514.5, 431.0, 585.7, 545.5, 506.0, 583.6, 395.0, 220.9, 75.97,
        35.26, 11.92, 5.557, 4.892, 25.18, 15.99, 9.977, 4.664, 2.588,
        0.918, 0.4687, 0.2126, 0.1389, 0.1071, 0.08976, 0.07105, 0.0607,
        0.05268, 0.04779, 0.04244, 0.03858, 0.03774, 0.038, 0.0387,
        0.04067, 0.04288, 0.04809, 0.05226,
      }, {
        10540.0, 8091.0, 6254.0, 7196.0, 6635.0, 6118.0, 6487.0, 5069.0,
        4986.0, 5214.0, 2755.0, 1038.0, 506.1, 286.5, 181.6, 480.2,
        473.9, 398.4, 533.3, 497.6, 462.5, 531.7, 363.4, 205.2, 70.94,
        32.67, 10.72, 4.803, 4.192, 7.833, 6.567, 4.942, 2.773, 1.667,
        0.6172, 0.3025, 0.1195, 0.0698, 0.05047, 0.04107, 0.03229,
        0.02811, 0.02503, 0.0232, 0.02152, 0.02131, 0.02233, 0.02359,
        0.02481, 0.02697, 0.02872, 0.03155, 0.03277,
      }
    },


    // Z = 60
    {
      57, {
        0.001, 0.0010025, 0.001005, 0.001005, 0.00114188, 0.0012974,
        0.0012974, 0.00134907, 0.0014028, 0.0014028, 0.0015, 0.0015753,
        0.0015753, 0.002, 0.003, 0.004, 0.005, 0.006, 0.0062079,
        0.0062079, 0.0064596, 0.0067215, 0.0067215, 0.0069208, 0.007126,
        0.007126, 0.008, 0.01, 0.015, 0.02, 0.03, 0.04, 0.0435689,
        0.0435689, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6,
        0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0,
        20.0,
      }, {
        6627.0, 6806.0, 6949.0, 8554.0, 7579.0, 5933.0, 6833.0, 6264.0,
        5744.0, 6093.0, 5273.0, 4738.0, 4951.0, 2878.0, 1093.0, 536.6,
        306.1, 192.7, 176.7, 489.2, 448.7, 401.3, 545.6, 513.5, 472.3,
        544.8, 409.4, 230.0, 79.25, 36.84, 12.47, 5.809, 4.642, 23.66,
        16.5, 10.33, 4.839, 2.687, 0.9522, 0.4844, 0.2178, 0.1412,
        0.1083, 0.0904, 0.07127, 0.06072, 0.05262, 0.04769, 0.04236,
        0.03856, 0.03776, 0.03806, 0.03879, 0.04081, 0.04304, 0.04832,
        0.05257,
      }, {
        6598.0, 6776.0, 6918.0, 8516.0, 7547.0, 5908.0, 6803.0, 6236.0,
        5719.0, 6066.0, 5250.0, 4717.0, 4929.0, 2864.0, 1084.0, 529.4,
        300.0, 187.4, 171.5, 448.4, 407.9, 369.4, 493.9, 461.4, 429.3,
        493.6, 373.9, 212.5, 73.76, 34.07, 11.21, 5.033, 3.962, 7.296,
        6.418, 4.928, 2.815, 1.706, 0.6376, 0.3133, 0.1236, 0.07177,
        0.05157, 0.04174, 0.03258, 0.02823, 0.02505, 0.02318, 0.02148,
        0.02129, 0.02233, 0.0236, 0.02483, 0.02701, 0.02876, 0.03159,
        0.03283,
      }
    },


    // Z = 61
    {
      60, {
        0.001, 0.00101336, 0.0010269, 0.0010269, 0.00103913, 0.0010515,
        0.0010515, 0.00119448, 0.0013569, 0.0013569, 0.00141299,
        0.0014714, 0.0014714, 0.0015, 0.001653, 0.001653, 0.002, 0.003,
        0.004, 0.005, 0.006, 0.0064593, 0.0064593, 0.00673036,
        0.0070128, 0.0070128, 0.00721737, 0.0074279, 0.0074279, 0.008,
        0.01, 0.015, 0.02, 0.03, 0.04, 0.045184, 0.045184, 0.05, 0.06,
        0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5,
        2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2056.0, 2007.0, 1959.0, 2729.0, 3894.0, 5830.0, 7242.0, 7232.0,
        5687.0, 6555.0, 5985.0, 5466.0, 5799.0, 5553.0, 4497.0, 4699.0,
        3048.0, 1162.0, 570.9, 326.0, 205.3, 170.2, 467.8, 422.6, 380.5,
        517.4, 488.4, 449.4, 518.4, 431.4, 244.0, 84.11, 39.16, 13.27,
        6.181, 4.491, 22.65, 17.35, 10.87, 5.107, 2.84, 1.005, 0.5098,
        0.2273, 0.1462, 0.1115, 0.09276, 0.0728, 0.06188, 0.0535,
        0.04849, 0.04308, 0.03925, 0.03849, 0.03882, 0.03959, 0.04169,
        0.04401, 0.04946, 0.0538,
      }, {
        2046.0, 1997.0, 1949.0, 2715.0, 3874.0, 5801.0, 7205.0, 7196.0,
        5660.0, 6524.0, 5956.0, 5439.0, 5770.0, 5526.0, 4474.0, 4676.0,
        3031.0, 1152.0, 563.3, 319.6, 199.8, 164.9, 426.7, 386.4, 348.7,
        465.7, 435.9, 406.2, 466.9, 390.8, 224.0, 77.99, 36.14, 11.93,
        5.366, 3.816, 6.93, 6.363, 4.989, 2.904, 1.776, 0.6699, 0.3302,
        0.1301, 0.07517, 0.05369, 0.04323, 0.03349, 0.02889, 0.02553,
        0.0236, 0.02184, 0.02166, 0.02273, 0.02405, 0.02532, 0.02754,
        0.02933, 0.03222, 0.03344,
      }
    },


    // Z = 62
    {
      60, {
        0.001, 0.00103933, 0.0010802, 0.0010802, 0.00109302, 0.001106,
        0.001106, 0.00125312, 0.0014198, 0.0014198, 0.0015, 0.0015407,
        0.0015407, 0.00162921, 0.0017228, 0.0017228, 0.002, 0.003,
        0.004, 0.005, 0.006, 0.0067162, 0.0067162, 0.00700767,
        0.0073118, 0.0073118, 0.0075213, 0.0077368, 0.0077368, 0.008,
        0.01, 0.015, 0.02, 0.03, 0.04, 0.0468342, 0.0468342, 0.05, 0.06,
        0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5,
        2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2107.0, 1963.0, 1827.0, 2635.0, 3837.0, 5613.0, 6852.0, 6641.0,
        5262.0, 6072.0, 5358.0, 5046.0, 5355.0, 4740.0, 4189.0, 4376.0,
        3120.0, 1193.0, 587.3, 335.6, 211.5, 158.7, 433.4, 389.8, 349.6,
        475.6, 450.1, 414.1, 477.7, 440.1, 249.9, 86.33, 40.25, 13.65,
        6.362, 4.208, 21.0, 17.74, 11.07, 5.212, 2.901, 1.027, 0.5192,
        0.2296, 0.1466, 0.1112, 0.09218, 0.07201, 0.06106, 0.05271,
        0.04773, 0.0424, 0.03868, 0.03797, 0.03833, 0.03912, 0.04124,
        0.04355, 0.04899, 0.05335,
      }, {
        2097.0, 1953.0, 1817.0, 2620.0, 3816.0, 5582.0, 6814.0, 6605.0,
        5234.0, 6039.0, 5329.0, 5018.0, 5325.0, 4714.0, 4165.0, 4352.0,
        3102.0, 1183.0, 579.5, 329.1, 205.9, 153.6, 393.3, 354.7, 318.9,
        425.5, 398.8, 372.1, 427.6, 395.1, 227.8, 79.72, 37.06, 12.27,
        5.532, 3.562, 6.378, 6.094, 4.872, 2.893, 1.786, 0.6806, 0.3366,
        0.1326, 0.0762, 0.05411, 0.04334, 0.03333, 0.02862, 0.02522,
        0.02325, 0.0215, 0.02133, 0.02241, 0.02372, 0.02498, 0.02719,
        0.02896, 0.0318, 0.03302,
      }
    },


    // Z = 63
    {
      59, {
        0.001, 0.00106344, 0.0011309, 0.0011309, 0.00114565, 0.0011606,
        0.0011606, 0.00131087, 0.0014806, 0.0014806, 0.0015, 0.0016139,
        0.0016139, 0.00170441, 0.0018, 0.0018, 0.002, 0.003, 0.004,
        0.005, 0.006, 0.0069769, 0.0069769, 0.00728997, 0.0076171,
        0.0076171, 0.008, 0.008052, 0.008052, 0.01, 0.015, 0.02, 0.03,
        0.04, 0.048519, 0.048519, 0.05, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0,
        8.0, 10.0, 15.0, 20.0,
      }, {
        2216.0, 1978.0, 1763.0, 2443.0, 3617.0, 5416.0, 6540.0, 6291.0,
        5031.0, 5810.0, 5624.0, 4771.0, 5064.0, 4492.0, 3980.0, 4158.0,
        3278.0, 1256.0, 619.3, 354.2, 223.4, 152.2, 412.7, 369.6, 330.2,
        449.2, 398.9, 392.1, 452.4, 262.9, 90.87, 42.42, 14.41, 6.716,
        4.051, 20.01, 18.5, 11.55, 5.455, 3.04, 1.076, 0.5425, 0.238,
        0.1509, 0.1139, 0.09404, 0.07312, 0.06186, 0.05331, 0.04822,
        0.04285, 0.03913, 0.03845, 0.03885, 0.03967, 0.04185, 0.04422,
        0.0498, 0.05423,
      }, {
        2206.0, 1968.0, 1753.0, 2428.0, 3595.0, 5382.0, 6498.0, 6253.0,
        5001.0, 5774.0, 5590.0, 4742.0, 5034.0, 4465.0, 3955.0, 4132.0,
        3258.0, 1245.0, 611.1, 347.4, 217.5, 147.1, 372.5, 334.6, 299.8,
        399.3, 356.0, 350.2, 402.4, 237.8, 83.52, 38.95, 12.95, 5.848,
        3.414, 6.036, 5.927, 4.863, 2.953, 1.841, 0.7088, 0.3518,
        0.1385, 0.07927, 0.05598, 0.04462, 0.03406, 0.02912, 0.02557,
        0.02353, 0.02173, 0.02158, 0.02269, 0.02404, 0.02532, 0.02758,
        0.02938, 0.03228, 0.03351,
      }
    },


    // Z = 64
    {
      59, {
        0.001, 0.00108867, 0.0011852, 0.0011852, 0.00120109, 0.0012172,
        0.0012172, 0.0015, 0.001544, 0.001544, 0.00161454, 0.0016883,
        0.0016883, 0.00178195, 0.0018808, 0.0018808, 0.002, 0.003,
        0.004, 0.005, 0.006, 0.0072428, 0.0072428, 0.00757876,
        0.0079303, 0.0079303, 0.008, 0.0083756, 0.0083756, 0.01, 0.015,
        0.02, 0.03, 0.04, 0.05, 0.0502391, 0.0502391, 0.06, 0.08, 0.1,
        0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0,
        4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2291.0, 1960.0, 1668.0, 1931.0, 2527.0, 3961.0, 4764.0, 5041.0,
        4701.0, 5432.0, 4900.0, 4421.0, 4694.0, 4164.0, 3691.0, 3854.0,
        3360.0, 1292.0, 638.0, 365.3, 230.5, 142.9, 384.4, 342.7, 304.9,
        414.9, 406.8, 363.1, 419.0, 269.3, 93.35, 43.63, 14.84, 6.92,
        3.859, 3.812, 18.64, 11.75, 5.573, 3.109, 1.1, 0.5534, 0.241,
        0.1517, 0.1139, 0.09371, 0.07252, 0.0612, 0.05262, 0.04759,
        0.04228, 0.03865, 0.03802, 0.03844, 0.03928, 0.04147, 0.04384,
        0.04943, 0.05385,
      }, {
        2281.0, 1949.0, 1658.0, 1919.0, 2510.0, 3934.0, 4731.0, 5008.0,
        4670.0, 5395.0, 4867.0, 4391.0, 4662.0, 4136.0, 3666.0, 3829.0,
        3337.0, 1280.0, 629.6, 358.4, 224.6, 137.9, 345.2, 308.8, 275.5,
        366.5, 359.5, 322.3, 370.2, 241.6, 85.38, 39.94, 13.33, 6.033,
        3.242, 3.199, 5.585, 4.722, 2.937, 1.849, 0.7197, 0.3584,
        0.1409, 0.08039, 0.0565, 0.04483, 0.03399, 0.02893, 0.0253,
        0.02325, 0.02144, 0.02129, 0.02241, 0.02374, 0.02502, 0.02725,
        0.02902, 0.03188, 0.03306,
      }
    },


    // Z = 65
    {
      59, {
        0.001, 0.00111409, 0.0012412, 0.0012412, 0.00125799, 0.001275,
        0.001275, 0.0015, 0.0016113, 0.0016113, 0.00168769, 0.0017677,
        0.0017677, 0.00186493, 0.0019675, 0.0019675, 0.002, 0.003,
        0.004, 0.005, 0.006, 0.007514, 0.007514, 0.008, 0.0082516,
        0.0082516, 0.00847673, 0.008708, 0.008708, 0.01, 0.015, 0.02,
        0.03, 0.04, 0.05, 0.0519957, 0.0519957, 0.06, 0.08, 0.1, 0.15,
        0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0,
        5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2396.0, 1956.0, 1591.0, 2337.0, 3515.0, 5023.0, 5928.0, 5314.0,
        4464.0, 5164.0, 4635.0, 4166.0, 4424.0, 3929.0, 3482.0, 3638.0,
        3507.0, 1354.0, 669.7, 383.8, 242.3, 136.8, 365.8, 313.3, 288.1,
        392.5, 368.4, 344.1, 397.0, 281.5, 98.02, 45.88, 15.63, 7.288,
        4.064, 3.672, 17.77, 12.23, 5.826, 3.253, 1.151, 0.5775, 0.2498,
        0.1562, 0.1167, 0.09562, 0.07365, 0.06196, 0.05321, 0.04808,
        0.04272, 0.03908, 0.03848, 0.03894, 0.03981, 0.04206, 0.04449,
        0.05017, 0.0547,
      }, {
        2386.0, 1946.0, 1581.0, 2321.0, 3488.0, 4984.0, 5881.0, 5274.0,
        4431.0, 5125.0, 4601.0, 4135.0, 4391.0, 3899.0, 3456.0, 3611.0,
        3481.0, 1342.0, 660.7, 376.5, 236.2, 131.8, 326.6, 281.0, 259.0,
        344.3, 323.9, 303.3, 348.3, 250.2, 89.14, 41.84, 14.02, 6.357,
        3.422, 3.068, 5.291, 4.673, 2.986, 1.899, 0.7474, 0.3736, 0.147,
        0.08354, 0.05842, 0.04614, 0.03474, 0.02942, 0.02565, 0.02352,
        0.02166, 0.02151, 0.02266, 0.02402, 0.02532, 0.02758, 0.02937,
        0.03224, 0.03343,
      }
    },


    // Z = 66
    {
      58, {
        0.001, 0.00113794, 0.0012949, 0.0012949, 0.00131356, 0.0013325,
        0.0013325, 0.0015, 0.0016756, 0.0016756, 0.00175674, 0.0018418,
        0.0018418, 0.002, 0.0020468, 0.0020468, 0.003, 0.004, 0.005,
        0.006, 0.0077901, 0.0077901, 0.008, 0.0085806, 0.0085806,
        0.00881013, 0.0090458, 0.0090458, 0.01, 0.015, 0.02, 0.03, 0.04,
        0.05, 0.0537885, 0.0537885, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0,
        8.0, 10.0, 15.0, 20.0,
      }, {
        2494.0, 1953.0, 1521.0, 2173.0, 3293.0, 4745.0, 5551.0, 5550.0,
        4229.0, 4896.0, 4382.0, 3924.0, 4168.0, 3467.0, 3286.0, 3433.0,
        1405.0, 695.3, 398.8, 252.0, 129.9, 343.7, 326.9, 269.5, 367.2,
        345.2, 322.8, 372.5, 290.2, 101.6, 47.65, 16.25, 7.582, 4.227,
        3.5, 16.76, 12.59, 6.012, 3.36, 1.189, 0.5953, 0.2558, 0.159,
        0.1181, 0.09644, 0.07393, 0.06204, 0.05318, 0.04801, 0.04265,
        0.03905, 0.03848, 0.03896, 0.03986, 0.04214, 0.0446, 0.05033,
        0.05492,
      }, {
        2483.0, 1943.0, 1511.0, 2156.0, 3265.0, 4704.0, 5502.0, 5503.0,
        4194.0, 4855.0, 4345.0, 3892.0, 4133.0, 3438.0, 3259.0, 3404.0,
        1391.0, 685.8, 391.3, 245.6, 124.9, 305.1, 290.6, 241.0, 319.8,
        301.3, 282.6, 324.5, 255.4, 91.87, 43.28, 14.56, 6.616, 3.566,
        2.911, 4.961, 4.554, 2.993, 1.925, 0.766, 0.3846, 0.1514,
        0.08578, 0.05971, 0.04694, 0.03509, 0.02958, 0.0257, 0.02352,
        0.02163, 0.02149, 0.02264, 0.02401, 0.02532, 0.02758, 0.02938,
        0.03223, 0.03342,
      }
    },


    // Z = 67
    {
      59, {
        0.001, 0.0011625, 0.0013514, 0.0013514, 0.0013713, 0.0013915,
        0.0013915, 0.0015, 0.0017412, 0.0017412, 0.00182975, 0.0019228,
        0.0019228, 0.002, 0.0021283, 0.0021283, 0.003, 0.004, 0.005,
        0.006, 0.008, 0.0080711, 0.0080711, 0.00848389, 0.0089178,
        0.0089178, 0.0091529, 0.0093942, 0.0093942, 0.01, 0.015, 0.02,
        0.03, 0.04, 0.05, 0.0556177, 0.0556177, 0.06, 0.08, 0.1, 0.15,
        0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0,
        5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2616.0, 1961.0, 1462.0, 2108.0, 3193.0, 4528.0, 5239.0, 5847.0,
        4038.0, 4677.0, 4161.0, 3703.0, 3934.0, 3590.0, 3127.0, 3266.0,
        1465.0, 726.4, 417.0, 263.6, 127.1, 124.2, 328.3, 288.9, 254.0,
        346.3, 325.9, 305.1, 352.0, 301.2, 106.0, 49.8, 17.01, 7.94,
        4.425, 3.361, 15.92, 13.09, 6.244, 3.492, 1.236, 0.6178, 0.2639,
        0.1629, 0.1205, 0.098, 0.07477, 0.06257, 0.05351, 0.04828,
        0.04289, 0.0393, 0.03877, 0.03927, 0.0402, 0.04253, 0.04504,
        0.05088, 0.05551,
      }, {
        2605.0, 1951.0, 1452.0, 2091.0, 3164.0, 4485.0, 5187.0, 5790.0,
        4001.0, 4633.0, 4122.0, 3669.0, 3898.0, 3558.0, 3099.0, 3236.0,
        1450.0, 716.4, 409.1, 257.0, 122.0, 119.2, 289.7, 256.0, 226.0,
        299.4, 282.5, 265.2, 304.4, 262.3, 95.21, 45.04, 15.21, 6.928,
        3.74, 2.783, 4.697, 4.466, 3.015, 1.961, 0.7901, 0.3984, 0.157,
        0.08869, 0.06145, 0.04811, 0.03571, 0.02997, 0.02593, 0.02369,
        0.02175, 0.02161, 0.02279, 0.02417, 0.02549, 0.02777, 0.02958,
        0.03245, 0.03363,
      }
    },


    // Z = 68
    {
      59, {
        0.001, 0.00118714, 0.0014093, 0.0014093, 0.00143113, 0.0014533,
        0.0014533, 0.0015, 0.0018118, 0.0018118, 0.002, 0.0020058,
        0.0020058, 0.00210376, 0.0022065, 0.0022065, 0.003, 0.004,
        0.005, 0.006, 0.008, 0.0083579, 0.0083579, 0.00879944,
        0.0092643, 0.0092643, 0.00950468, 0.0097513, 0.0097513, 0.01,
        0.015, 0.02, 0.03, 0.04, 0.05, 0.0574855, 0.0574855, 0.06, 0.08,
        0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0,
        3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        2748.0, 1978.0, 1408.0, 2069.0, 3170.0, 4339.0, 4952.0, 6069.0,
        3841.0, 4452.0, 3523.0, 3499.0, 3717.0, 3337.0, 2995.0, 3127.0,
        1526.0, 758.7, 435.9, 275.7, 133.0, 119.0, 312.6, 279.9, 239.7,
        327.1, 308.0, 288.6, 333.0, 312.9, 110.6, 52.04, 17.8, 8.315,
        4.634, 3.232, 15.14, 13.62, 6.478, 3.628, 1.285, 0.6415, 0.2724,
        0.1672, 0.123, 0.09968, 0.07569, 0.06317, 0.05392, 0.0486,
        0.04317, 0.03958, 0.03907, 0.03961, 0.04056, 0.04294, 0.04549,
        0.05141, 0.05613,
      }, {
        2737.0, 1967.0, 1397.0, 2051.0, 3138.0, 4293.0, 4898.0, 6003.0,
        3802.0, 4406.0, 3488.0, 3464.0, 3680.0, 3304.0, 2965.0, 3096.0,
        1510.0, 748.0, 427.6, 268.8, 127.8, 114.0, 274.1, 241.2, 212.1,
        280.6, 265.0, 249.1, 285.8, 269.4, 98.6, 46.83, 15.88, 7.253,
        3.922, 2.664, 4.442, 4.349, 3.028, 1.995, 0.8141, 0.4125,
        0.1629, 0.09173, 0.06329, 0.04934, 0.03638, 0.0304, 0.0262,
        0.02388, 0.0219, 0.02176, 0.02295, 0.02435, 0.02569, 0.02799,
        0.02981, 0.03268, 0.03387,
      }
    },


    // Z = 69
    {
      58, {
        0.001, 0.00121149, 0.0014677, 0.0014677, 0.0015, 0.0015146, 0.0015146,
        0.00168946, 0.0018845, 0.0018845, 0.002, 0.0020898, 0.0020898,
        0.00219562, 0.0023068, 0.0023068, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.008648, 0.008648, 0.00911959, 0.0096169, 0.0096169,
        0.01, 0.0101157, 0.0101157, 0.015, 0.02, 0.03, 0.04, 0.05,
        0.0593896, 0.0593896, 0.06, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        2899.0, 1995.0, 1362.0, 1919.0, 3937.0, 4145.0, 4830.0, 4446.0,
        3668.0, 4254.0, 3686.0, 3326.0, 3534.0, 3160.0, 2821.0, 2945.0,
        1594.0, 794.6, 456.9, 289.2, 139.7, 114.6, 299.1, 260.9, 227.4,
        310.1, 283.0, 274.5, 316.7, 115.7, 54.53, 18.68, 8.735, 4.867,
        3.122, 12.0, 14.09, 6.741, 3.78, 1.34, 0.6682, 0.2822, 0.1722,
        0.1261, 0.1018, 0.07693, 0.06404, 0.05455, 0.04912, 0.04362,
        0.04002, 0.03954, 0.0401, 0.04108, 0.04352, 0.04612, 0.05216,
        0.05698,
      }, {
        2888.0, 1984.0, 1352.0, 1901.0, 3892.0, 4098.0, 4773.0, 4395.0,
        3627.0, 4206.0, 3646.0, 3290.0, 3495.0, 3125.0, 2791.0, 2914.0,
        1576.0, 783.1, 448.2, 282.0, 134.2, 109.6, 260.6, 228.4, 200.1,
        264.0, 242.0, 235.1, 269.7, 102.3, 48.82, 16.63, 7.615, 4.124,
        2.562, 4.721, 4.203, 3.044, 2.033, 0.841, 0.4283, 0.1695,
        0.09522, 0.06544, 0.05081, 0.03721, 0.03095, 0.02658, 0.02417,
        0.02214, 0.02199, 0.02321, 0.02463, 0.02598, 0.02832, 0.03016,
        0.03305, 0.03424,
      }
    },


    // Z = 70
    {
      58, {
        0.001, 0.0015, 0.0015278, 0.0015278, 0.00155186, 0.0015763, 0.0015763,
        0.00175313, 0.0019498, 0.0019498, 0.002, 0.002173, 0.002173,
        0.00228278, 0.0023981, 0.0023981, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.0089436, 0.0089436, 0.00944675, 0.0099782, 0.0099782,
        0.01, 0.0104864, 0.0104864, 0.015, 0.02, 0.03, 0.04, 0.05, 0.06,
        0.0613323, 0.0613323, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6,
        0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0,
        20.0,
      }, {
        3017.0, 1350.0, 1300.0, 1965.0, 2949.0, 3908.0, 4540.0, 4200.0,
        3495.0, 4054.0, 3797.0, 3128.0, 3323.0, 2970.0, 2652.0, 2769.0,
        1640.0, 819.3, 471.7, 298.8, 144.4, 108.8, 282.4, 251.1, 212.8,
        290.6, 289.3, 257.7, 297.3, 119.3, 56.28, 19.32, 9.04, 5.038,
        3.147, 2.975, 13.65, 6.909, 3.881, 1.378, 0.686, 0.2882, 0.1749,
        0.1274, 0.1025, 0.0771, 0.06397, 0.05439, 0.04895, 0.04347,
        0.03989, 0.03944, 0.04002, 0.04101, 0.04347, 0.04609, 0.05218,
        0.057,
      }, {
        3006.0, 1340.0, 1289.0, 1944.0, 2914.0, 3859.0, 4482.0, 4146.0,
        3453.0, 4004.0, 3751.0, 3091.0, 3284.0, 2935.0, 2621.0, 2737.0,
        1620.0, 807.2, 462.6, 291.3, 138.9, 103.9, 244.4, 213.4, 186.2,
        245.5, 244.4, 218.9, 251.2, 104.7, 50.12, 17.16, 7.875, 4.273,
        2.584, 2.432, 3.964, 3.008, 2.038, 0.8554, 0.4381, 0.1737,
        0.09741, 0.06671, 0.05159, 0.03754, 0.03107, 0.02658, 0.02413,
        0.02208, 0.02191, 0.02314, 0.02457, 0.02593, 0.02826, 0.0301,
        0.03302, 0.03419,
      }
    },


    // Z = 71
    {
      58, {
        0.001, 0.0015, 0.0015885, 0.0015885, 0.00161375, 0.0016394, 0.0016394,
        0.002, 0.0020236, 0.0020236, 0.00214019, 0.0022635, 0.0022635,
        0.00237462, 0.0024912, 0.0024912, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.0092441, 0.0092441, 0.01, 0.0103486, 0.0103486,
        0.0106063, 0.0108704, 0.0108704, 0.015, 0.02, 0.03, 0.04, 0.05,
        0.06, 0.0633138, 0.0633138, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        3187.0, 1424.0, 1264.0, 1597.0, 2302.0, 3442.0, 3935.0, 3452.0,
        3352.0, 3890.0, 3397.0, 2969.0, 3155.0, 2826.0, 2531.0, 2640.0,
        1710.0, 856.0, 493.4, 312.9, 151.3, 104.9, 270.3, 221.1, 201.9,
        275.7, 267.6, 245.0, 282.6, 124.7, 58.81, 20.23, 9.472, 5.279,
        3.297, 2.874, 13.05, 7.161, 4.033, 1.433, 0.713, 0.2981, 0.1799,
        0.1305, 0.1046, 0.07829, 0.06478, 0.05496, 0.04941, 0.04385,
        0.04028, 0.03985, 0.04045, 0.04147, 0.04398, 0.04664, 0.05282,
        0.05773,
      }, {
        3175.0, 1413.0, 1253.0, 1581.0, 2273.0, 3396.0, 3881.0, 3406.0,
        3307.0, 3838.0, 3352.0, 2931.0, 3115.0, 2790.0, 2499.0, 2607.0,
        1688.0, 843.0, 483.9, 305.1, 145.6, 99.94, 232.4, 191.6, 175.6,
        231.0, 219.0, 206.5, 236.9, 108.4, 52.06, 17.91, 8.242, 4.48,
        2.712, 2.339, 3.77, 3.0, 2.067, 0.8805, 0.4534, 0.1803, 0.1009,
        0.06886, 0.05306, 0.03836, 0.03161, 0.02695, 0.02441, 0.02228,
        0.02212, 0.02337, 0.02481, 0.02618, 0.02853, 0.03038, 0.03329,
        0.03446,
      }
    },


    // Z = 72
    {
      58, {
        0.001, 0.0015, 0.0016617, 0.0016617, 0.00168883, 0.0017164, 0.0017164,
        0.002, 0.0021076, 0.0021076, 0.00223278, 0.0023654, 0.0023654,
        0.00248036, 0.0026009, 0.0026009, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.0095607, 0.0095607, 0.01, 0.0107394, 0.0107394,
        0.0110018, 0.0112707, 0.0112707, 0.015, 0.02, 0.03, 0.04, 0.05,
        0.06, 0.0653508, 0.0653508, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        3335.0, 1489.0, 1203.0, 1505.0, 2189.0, 3273.0, 3682.0, 3598.0,
        3152.0, 3659.0, 3182.0, 2769.0, 2943.0, 2637.0, 2363.0, 2465.0,
        1768.0, 885.9, 511.3, 324.4, 157.1, 99.97, 255.7, 230.1, 189.4,
        258.9, 252.2, 230.6, 265.9, 129.0, 60.87, 20.98, 9.828, 5.478,
        3.42, 2.751, 12.37, 7.352, 4.154, 1.477, 0.7339, 0.3054, 0.1834,
        0.1324, 0.1058, 0.0788, 0.06502, 0.05505, 0.04944, 0.04387,
        0.0403, 0.03989, 0.04052, 0.04155, 0.04409, 0.04677, 0.05301,
        0.05796,
      }, {
        3324.0, 1478.0, 1192.0, 1489.0, 2160.0, 3225.0, 3628.0, 3546.0,
        3108.0, 3607.0, 3137.0, 2731.0, 2903.0, 2601.0, 2331.0, 2432.0,
        1745.0, 872.0, 501.3, 316.4, 151.2, 95.1, 218.4, 197.5, 163.9,
        215.1, 204.2, 192.8, 221.2, 111.1, 53.53, 18.51, 8.542, 4.651,
        2.82, 2.228, 3.557, 2.958, 2.075, 0.8973, 0.4645, 0.1853,
        0.1035, 0.07044, 0.05409, 0.03886, 0.03188, 0.02708, 0.02447,
        0.0223, 0.02212, 0.02338, 0.02483, 0.0262, 0.02855, 0.0304,
        0.0333, 0.03445,
      }
    },


    // Z = 73
    {
      58, {
        0.001, 0.0015, 0.0017351, 0.0017351, 0.00176391, 0.0017932, 0.0017932,
        0.002, 0.002194, 0.002194, 0.0023273, 0.0024687, 0.0024687,
        0.00258558, 0.002708, 0.002708, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.0098811, 0.0098811, 0.01, 0.0111361, 0.0111361,
        0.0114055, 0.0116815, 0.0116815, 0.015, 0.02, 0.03, 0.04, 0.05,
        0.06, 0.0674164, 0.0674164, 0.08, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        3510.0, 1566.0, 1154.0, 1417.0, 2053.0, 3082.0, 3421.0, 3771.0,
        2985.0, 3464.0, 3003.0, 2604.0, 2768.0, 2486.0, 2233.0, 2329.0,
        1838.0, 922.2, 532.8, 338.2, 163.9, 95.99, 244.3, 237.9, 179.1,
        244.9, 239.3, 218.2, 251.8, 134.0, 63.34, 21.87, 10.25, 5.717,
        3.569, 2.652, 11.8, 7.587, 4.302, 1.531, 0.7598, 0.3149, 0.1881,
        0.1352, 0.1076, 0.07981, 0.06567, 0.05545, 0.04977, 0.04413,
        0.04057, 0.04018, 0.04082, 0.04188, 0.04446, 0.04717, 0.0535,
        0.05852,
      }, {
        3498.0, 1555.0, 1144.0, 1401.0, 2024.0, 3034.0, 3367.0, 3711.0,
        2939.0, 3411.0, 2957.0, 2566.0, 2727.0, 2449.0, 2201.0, 2296.0,
        1812.0, 907.3, 522.3, 329.9, 157.9, 91.17, 207.3, 202.1, 154.0,
        201.8, 191.7, 181.1, 207.7, 114.3, 55.32, 19.23, 8.899, 4.854,
        2.947, 2.139, 3.379, 2.924, 2.092, 0.9188, 0.4784, 0.1915,
        0.1069, 0.07248, 0.05545, 0.0396, 0.03235, 0.02736, 0.02468,
        0.02245, 0.02227, 0.02354, 0.02499, 0.02638, 0.02874, 0.03058,
        0.03349, 0.03464,
      }
    },


    // Z = 74
    {
      59, {
        0.001, 0.0015, 0.0018092, 0.0018092, 0.00184014, 0.0018716, 0.0018716,
        0.002, 0.002281, 0.002281, 0.0024235, 0.0025749, 0.0025749,
        0.00269447, 0.0028196, 0.0028196, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.01, 0.0102068, 0.0102068, 0.0108548, 0.011544,
        0.011544, 0.0118186, 0.0120998, 0.0120998, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.069525, 0.069525, 0.08, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0,
        8.0, 10.0, 15.0, 20.0,
      }, {
        3683.0, 1643.0, 1108.0, 1327.0, 1911.0, 2901.0, 3170.0, 3922.0,
        2828.0, 3279.0, 2833.0, 2445.0, 2599.0, 2339.0, 2104.0, 2194.0,
        1902.0, 956.4, 553.4, 351.4, 170.5, 96.91, 92.01, 233.4, 198.3,
        168.9, 231.2, 226.8, 206.5, 238.2, 138.9, 65.73, 22.73, 10.67,
        5.949, 3.713, 2.552, 11.23, 7.81, 4.438, 1.581, 0.7844, 0.3238,
        0.1925, 0.1378, 0.1093, 0.08066, 0.06618, 0.05577, 0.05,
        0.04433, 0.04075, 0.04038, 0.04103, 0.0421, 0.04472, 0.04747,
        0.05384, 0.05893,
      }, {
        3671.0, 1632.0, 1097.0, 1311.0, 1883.0, 2853.0, 3116.0, 3853.0,
        2781.0, 3226.0, 2786.0, 2407.0, 2558.0, 2301.0, 2071.0, 2160.0,
        1873.0, 940.5, 542.3, 342.8, 164.3, 92.04, 87.24, 196.6, 168.4,
        144.4, 188.9, 179.7, 169.9, 194.8, 117.2, 56.97, 19.91, 9.24,
        5.05, 3.07, 2.049, 3.212, 2.879, 2.1, 0.9378, 0.4913, 0.1973,
        0.11, 0.0744, 0.05673, 0.04028, 0.03276, 0.02761, 0.02484,
        0.02256, 0.02236, 0.02363, 0.0251, 0.02649, 0.02886, 0.03072,
        0.0336, 0.03475,
      }
    },


    // Z = 75
    {
      59, {
        0.001, 0.0015, 0.0018224, 0.0018224, 0.00188459, 0.0019489, 0.0019489,
        0.002, 0.0023673, 0.0023673, 0.00251955, 0.0026816, 0.0026816,
        0.00280386, 0.0029317, 0.0029317, 0.003, 0.004, 0.005, 0.006,
        0.008, 0.01, 0.0105353, 0.0105353, 0.0112245, 0.0119587,
        0.0119587, 0.0122394, 0.0125267, 0.0125267, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.0716764, 0.0716764, 0.08, 0.1, 0.15, 0.2,
        0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0,
        6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        3872.0, 1729.0, 1147.0, 1152.0, 1251.0, 2689.0, 3802.0, 3773.0,
        2696.0, 3124.0, 2686.0, 2309.0, 2454.0, 2210.0, 1992.0, 2078.0,
        1972.0, 994.3, 575.9, 366.0, 177.8, 101.1, 88.58, 223.4, 196.4,
        160.0, 219.1, 207.9, 196.1, 226.2, 144.0, 68.35, 23.67, 11.12,
        6.206, 3.872, 2.464, 10.73, 8.069, 4.587, 1.637, 0.8119, 0.3339,
        0.1976, 0.1409, 0.1114, 0.08179, 0.06688, 0.05627, 0.05038,
        0.04463, 0.04105, 0.04068, 0.04137, 0.04246, 0.04511, 0.0479,
        0.05437, 0.05953,
      }, {
        3860.0, 1717.0, 1135.0, 1141.0, 1235.0, 2642.0, 3730.0, 3702.0,
        2648.0, 3069.0, 2639.0, 2270.0, 2412.0, 2174.0, 1959.0, 2043.0,
        1939.0, 977.1, 564.2, 356.9, 171.4, 96.1, 83.86, 186.9, 159.4,
        136.0, 177.6, 169.0, 160.0, 183.4, 120.1, 58.76, 20.64, 9.614,
        5.265, 3.205, 1.97, 3.058, 2.832, 2.107, 0.958, 0.5054, 0.2038,
        0.1135, 0.07658, 0.05822, 0.04109, 0.03327, 0.02794, 0.02508,
        0.02273, 0.02251, 0.0238, 0.02527, 0.02667, 0.02906, 0.03092,
        0.03382, 0.03496,
      }
    },


    // Z = 76
    {
      58, {
        0.001, 0.0015, 0.0019601, 0.0019601, 0.002, 0.0020308, 0.0020308,
        0.00223385, 0.0024572, 0.0024572, 0.00261935, 0.0027922,
        0.0027922, 0.003, 0.0030485, 0.0030485, 0.004, 0.005, 0.006,
        0.008, 0.01, 0.0108709, 0.0108709, 0.0116033, 0.012385,
        0.012385, 0.0126731, 0.012968, 0.012968, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.0738708, 0.0738708, 0.08, 0.1, 0.15, 0.2,
        0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0,
        6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        4032.0, 1801.0, 1023.0, 2003.0, 2218.0, 2622.0, 2864.0, 2852.0,
        2546.0, 2948.0, 2524.0, 2161.0, 2296.0, 1938.0, 1869.0, 1949.0,
        1023.0, 593.6, 377.6, 183.6, 104.5, 84.58, 212.1, 186.0, 150.3,
        206.0, 195.6, 184.6, 212.9, 147.8, 70.39, 24.43, 11.49, 6.414,
        4.002, 2.36, 10.16, 8.29, 4.696, 1.68, 0.8327, 0.3414, 0.2011,
        0.1428, 0.1125, 0.08224, 0.06705, 0.05625, 0.05034, 0.04458,
        0.041, 0.04065, 0.04134, 0.04244, 0.04511, 0.04791, 0.05442,
        0.05956,
      }, {
        4019.0, 1790.0, 1012.0, 1969.0, 2179.0, 2573.0, 2810.0, 2797.0,
        2498.0, 2893.0, 2478.0, 2123.0, 2255.0, 1904.0, 1836.0, 1915.0,
        1004.0, 581.3, 368.2, 177.0, 99.4, 79.94, 176.2, 149.5, 127.1,
        165.4, 157.6, 149.4, 171.1, 121.8, 59.99, 21.2, 9.907, 5.437,
        3.314, 1.879, 2.888, 2.756, 2.092, 0.9696, 0.515, 0.2085,
        0.1161, 0.07813, 0.05923, 0.04157, 0.03351, 0.02802, 0.02511,
        0.02272, 0.02248, 0.02376, 0.02523, 0.02663, 0.02901, 0.03086,
        0.03375, 0.03486,
      }
    },


    // Z = 77
    {
      59, {
        0.001, 0.0015, 0.002, 0.0020404, 0.0020404, 0.0020779, 0.0021161,
        0.0021161, 0.00232326, 0.0025507, 0.0025507, 0.00272382,
        0.0029087, 0.0029087, 0.003, 0.0031737, 0.0031737, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.0112154, 0.0112154, 0.0119928, 0.0128241,
        0.0128241, 0.0131179, 0.0134185, 0.0134185, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.076111, 0.076111, 0.08, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0,
        8.0, 10.0, 15.0, 20.0,
      }, {
        4243.0, 1898.0, 1032.0, 988.2, 1066.0, 1492.0, 2478.0, 2608.0, 2621.0,
        2424.0, 2800.0, 2393.0, 2041.0, 2166.0, 2011.0, 1767.0, 1843.0,
        1063.0, 617.8, 393.5, 191.4, 109.0, 81.55, 203.8, 170.2, 142.6,
        195.9, 193.3, 175.6, 202.5, 153.0, 73.17, 25.46, 11.99, 6.693,
        4.176, 2.284, 9.728, 8.585, 4.855, 1.74, 0.8628, 0.3525, 0.2068,
        0.1463, 0.1149, 0.08357, 0.06794, 0.05688, 0.05083, 0.04501,
        0.04139, 0.04104, 0.04174, 0.04286, 0.04558, 0.04844, 0.05502,
        0.06024,
      }, {
        4230.0, 1886.0, 1021.0, 977.1, 1053.0, 1466.0, 2429.0, 2556.0, 2566.0,
        2376.0, 2744.0, 2345.0, 2002.0, 2125.0, 1973.0, 1734.0, 1809.0,
        1043.0, 604.7, 383.7, 184.6, 103.8, 76.95, 168.1, 141.8, 119.9,
        155.9, 148.6, 140.9, 161.3, 124.4, 61.79, 21.98, 10.3, 5.667,
        3.46, 1.81, 2.758, 2.694, 2.093, 0.9903, 0.529, 0.2155, 0.1199,
        0.08054, 0.06088, 0.0425, 0.03412, 0.02843, 0.02541, 0.02295,
        0.02269, 0.02398, 0.02546, 0.02687, 0.02927, 0.03115, 0.03405,
        0.03516,
      }
    },


    // Z = 78
    {
      59, {
        0.001, 0.0015, 0.002, 0.0021216, 0.0021216, 0.00216138, 0.0022019,
        0.0022019, 0.00241348, 0.0026454, 0.0026454, 0.003, 0.0030265,
        0.0030265, 0.00315838, 0.003296, 0.003296, 0.004, 0.005, 0.006,
        0.008, 0.01, 0.0115637, 0.0115637, 0.0123887, 0.0132726,
        0.0132726, 0.0135729, 0.0138799, 0.0138799, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.0783948, 0.0783948, 0.08, 0.1, 0.15, 0.2,
        0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0,
        6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        4433.0, 1986.0, 1081.0, 950.6, 1034.0, 1431.0, 2341.0, 2487.0, 2512.0,
        2307.0, 2663.0, 1965.0, 1923.0, 2041.0, 1844.0, 1671.0, 1742.0,
        1100.0, 640.2, 408.3, 198.7, 113.2, 78.44, 194.6, 161.8, 134.9,
        185.3, 184.8, 166.6, 192.1, 157.8, 75.74, 26.41, 12.45, 6.954,
        4.339, 2.203, 9.301, 8.731, 4.993, 1.795, 0.8896, 0.3625,
        0.2118, 0.1492, 0.1168, 0.08456, 0.06857, 0.05727, 0.05112,
        0.04522, 0.0416, 0.04124, 0.04196, 0.0431, 0.04584, 0.04872,
        0.05537, 0.06064,
      }, {
        4421.0, 1974.0, 1069.0, 939.3, 1021.0, 1405.0, 2292.0, 2434.0, 2457.0,
        2258.0, 2606.0, 1925.0, 1884.0, 2000.0, 1806.0, 1637.0, 1707.0,
        1078.0, 626.2, 397.9, 191.7, 107.8, 73.88, 159.4, 133.9, 112.7,
        146.3, 139.5, 132.5, 151.7, 126.5, 63.33, 22.68, 10.67, 5.879,
        3.595, 1.738, 2.627, 2.592, 2.081, 1.006, 0.5413, 0.2216,
        0.1233, 0.08265, 0.0623, 0.04325, 0.03459, 0.02871, 0.0256,
        0.02307, 0.02279, 0.02408, 0.02557, 0.02699, 0.02939, 0.03127,
        0.03418, 0.03529,
      }
    },


    // Z = 79
    {
      59, {
        0.001, 0.0015, 0.002, 0.0022057, 0.0022057, 0.00224799, 0.0022911,
        0.0022911, 0.00250689, 0.002743, 0.002743, 0.003, 0.0031478,
        0.0031478, 0.00328343, 0.0034249, 0.0034249, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.0119187, 0.0119187, 0.012794, 0.0137336,
        0.0137336, 0.0140398, 0.0143528, 0.0143528, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.08, 0.0807249, 0.0807249, 0.1, 0.15, 0.2,
        0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0,
        6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        4652.0, 2089.0, 1137.0, 918.7, 997.1, 1386.0, 2258.0, 2389.0, 2380.0,
        2203.0, 2541.0, 2049.0, 1822.0, 1933.0, 1748.0, 1585.0, 1652.0,
        1144.0, 666.1, 425.3, 207.2, 118.1, 75.82, 187.0, 154.6, 128.3,
        176.4, 176.6, 158.8, 183.0, 163.7, 78.83, 27.52, 12.98, 7.256,
        4.528, 2.185, 2.137, 8.904, 5.158, 1.86, 0.9214, 0.3744, 0.218,
        0.153, 0.1194, 0.08603, 0.06953, 0.05794, 0.05167, 0.0457,
        0.04201, 0.04166, 0.04239, 0.04355, 0.04633, 0.04926, 0.05598,
        0.06136,
      }, {
        4639.0, 2076.0, 1125.0, 907.4, 983.6, 1360.0, 2208.0, 2336.0, 2325.0,
        2154.0, 2484.0, 2005.0, 1783.0, 1892.0, 1710.0, 1552.0, 1618.0,
        1120.0, 651.2, 414.3, 199.9, 112.6, 71.29, 152.1, 127.2, 106.6,
        137.9, 131.7, 125.2, 143.2, 129.4, 65.22, 23.49, 11.09, 6.124,
        3.751, 1.72, 1.678, 2.512, 2.074, 1.026, 0.5563, 0.2289, 0.1274,
        0.08523, 0.06409, 0.04427, 0.03525, 0.02915, 0.02593, 0.02333,
        0.02302, 0.02432, 0.02582, 0.02725, 0.02968, 0.03159, 0.0345,
        0.03565,
      }
    },


    // Z = 80
    {
      59, {
        0.001, 0.0015, 0.002, 0.0022949, 0.0022949, 0.00233947, 0.0023849,
        0.0023849, 0.00260577, 0.0028471, 0.0028471, 0.003, 0.0032785,
        0.0032785, 0.00341712, 0.0035616, 0.0035616, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.0122839, 0.0122839, 0.0132113, 0.0142087,
        0.0142087, 0.0145206, 0.0148393, 0.0148393, 0.015, 0.02, 0.03,
        0.04, 0.05, 0.06, 0.08, 0.0831023, 0.0831023, 0.1, 0.15, 0.2,
        0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0,
        6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        4830.0, 2174.0, 1184.0, 877.3, 992.5, 1407.0, 2151.0, 2316.0, 2257.0,
        2080.0, 2400.0, 2117.0, 1704.0, 1808.0, 1638.0, 1486.0, 1549.0,
        1179.0, 686.9, 438.7, 214.0, 122.1, 72.66, 178.0, 146.4, 120.9,
        166.3, 167.4, 150.1, 172.8, 168.1, 81.23, 28.41, 13.42, 7.504,
        4.683, 2.259, 2.055, 8.464, 5.279, 1.909, 0.9456, 0.3834,
        0.2224, 0.1555, 0.121, 0.08679, 0.06993, 0.05814, 0.05179,
        0.04575, 0.04207, 0.04172, 0.04246, 0.04362, 0.04643, 0.04937,
        0.05613, 0.06154,
      }, {
        4817.0, 2161.0, 1172.0, 866.0, 977.8, 1378.0, 2101.0, 2261.0, 2202.0,
        2030.0, 2343.0, 2069.0, 1666.0, 1767.0, 1600.0, 1454.0, 1515.0,
        1153.0, 671.0, 427.2, 206.5, 116.4, 68.21, 143.7, 119.7, 99.88,
        128.9, 123.3, 117.3, 134.0, 130.7, 66.47, 24.1, 11.41, 6.321,
        3.878, 1.782, 1.607, 2.384, 2.043, 1.036, 0.5661, 0.2342,
        0.1304, 0.08711, 0.06536, 0.04493, 0.03563, 0.02936, 0.02605,
        0.02339, 0.02307, 0.02436, 0.02586, 0.0273, 0.02975, 0.03165,
        0.0346, 0.03576,
      }
    },


    // Z = 81
    {
      58, {
        0.001, 0.0015, 0.002, 0.0023893, 0.0023893, 0.00243673, 0.0024851,
        0.0024851, 0.00271062, 0.0029566, 0.0029566, 0.003, 0.0034157,
        0.0034157, 0.00355698, 0.0037041, 0.0037041, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.0126575, 0.0126575, 0.0136396, 0.0146979,
        0.0146979, 0.015, 0.0153467, 0.0153467, 0.02, 0.03, 0.04, 0.05,
        0.06, 0.08, 0.0855304, 0.0855304, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5,
        0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0,
        15.0, 20.0,
      }, {
        5008.0, 2259.0, 1231.0, 835.2, 1136.0, 1582.0, 2049.0, 2364.0, 2207.0,
        1958.0, 2267.0, 2188.0, 1591.0, 1687.0, 1530.0, 1392.0, 1452.0,
        1212.0, 706.8, 451.8, 220.8, 126.0, 69.57, 169.3, 147.3, 113.9,
        157.2, 149.7, 141.6, 163.2, 83.61, 29.29, 13.85, 7.751, 4.838,
        2.332, 1.976, 8.046, 5.398, 1.957, 0.9696, 0.3923, 0.2267,
        0.158, 0.1226, 0.08751, 0.07031, 0.05832, 0.05187, 0.04581,
        0.0421, 0.04175, 0.04248, 0.04366, 0.04647, 0.04943, 0.05622,
        0.06162,
      }, {
        4995.0, 2247.0, 1219.0, 824.0, 1115.0, 1546.0, 1999.0, 2303.0, 2150.0,
        1910.0, 2210.0, 2135.0, 1553.0, 1647.0, 1495.0, 1360.0, 1419.0,
        1185.0, 690.0, 439.8, 213.1, 120.2, 65.21, 135.7, 112.6, 93.56,
        120.7, 115.4, 109.7, 125.4, 67.61, 24.67, 11.73, 6.513, 4.003,
        1.844, 1.539, 2.264, 2.007, 1.044, 0.5753, 0.2393, 0.1334,
        0.08897, 0.06661, 0.04557, 0.036, 0.02955, 0.02616, 0.02345,
        0.02309, 0.02437, 0.02587, 0.02731, 0.02975, 0.03167, 0.03463,
        0.03578,
      }
    },


    // Z = 82
    {
      58, {
        0.001, 0.0015, 0.002, 0.002484, 0.002484, 0.00253429, 0.0025856,
        0.0025856, 0.003, 0.0030664, 0.0030664, 0.0033013, 0.0035542,
        0.0035542, 0.00369948, 0.0038507, 0.0038507, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.0130352, 0.0130352, 0.015, 0.0152, 0.0152,
        0.0155269, 0.0158608, 0.0158608, 0.02, 0.03, 0.04, 0.05, 0.06,
        0.08, 0.0880045, 0.0880045, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6,
        0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0,
        20.0,
      }, {
        5210.0, 2356.0, 1285.0, 800.6, 1397.0, 1726.0, 1944.0, 2458.0, 1965.0,
        1857.0, 2146.0, 1796.0, 1496.0, 1585.0, 1442.0, 1311.0, 1368.0,
        1251.0, 730.4, 467.2, 228.7, 130.6, 67.01, 162.1, 111.6, 107.8,
        148.5, 141.6, 134.4, 154.8, 86.36, 30.32, 14.36, 8.041, 5.021,
        2.419, 1.91, 7.683, 5.549, 2.014, 0.9985, 0.4031, 0.2323,
        0.1614, 0.1248, 0.0887, 0.07102, 0.05876, 0.05222, 0.04606,
        0.04234, 0.04197, 0.04272, 0.04391, 0.04675, 0.04972, 0.05658,
        0.06206,
      }, {
        5197.0, 2344.0, 1274.0, 789.5, 1366.0, 1682.0, 1895.0, 2390.0, 1913.0,
        1808.0, 2090.0, 1748.0, 1459.0, 1546.0, 1405.0, 1279.0, 1335.0,
        1221.0, 712.4, 454.6, 220.7, 124.7, 62.7, 129.1, 91.0, 88.07,
        113.1, 108.3, 103.2, 118.0, 68.99, 25.36, 12.11, 6.74, 4.149,
        1.916, 1.482, 2.16, 1.976, 1.056, 0.587, 0.2455, 0.137, 0.09128,
        0.06819, 0.04644, 0.03654, 0.02988, 0.0264, 0.0236, 0.02322,
        0.02449, 0.026, 0.02744, 0.02989, 0.03181, 0.03478, 0.03595,
      }
    },


    // Z = 83
    {
      58, {
        0.001, 0.0015, 0.002, 0.0025796, 0.0025796, 0.00263305, 0.0026876,
        0.0026876, 0.003, 0.0031769, 0.0031769, 0.00342677, 0.0036963,
        0.0036963, 0.00384472, 0.0039991, 0.0039991, 0.004, 0.005,
        0.006, 0.008, 0.01, 0.0134186, 0.0134186, 0.015, 0.0157111,
        0.0157111, 0.0160457, 0.0163875, 0.0163875, 0.02, 0.03, 0.04,
        0.05, 0.06, 0.08, 0.0905259, 0.0905259, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0,
        8.0, 10.0, 15.0, 20.0,
      }, {
        5441.0, 2468.0, 1348.0, 772.4, 1777.0, 1850.0, 1852.0, 2576.0, 2053.0,
        1774.0, 2048.0, 1707.0, 1415.0, 1498.0, 1366.0, 1243.0, 1297.0,
        1296.0, 758.0, 485.5, 237.8, 136.0, 64.91, 156.0, 116.0, 102.7,
        141.6, 135.1, 128.2, 147.8, 89.52, 31.52, 14.95, 8.379, 5.233,
        2.522, 1.856, 7.38, 5.739, 2.082, 1.033, 0.4163, 0.2391, 0.1656,
        0.1277, 0.09036, 0.07214, 0.05955, 0.05285, 0.04659, 0.04279,
        0.04242, 0.04317, 0.04437, 0.04725, 0.05025, 0.05721, 0.06276,
      }, {
        5427.0, 2455.0, 1336.0, 761.2, 1731.0, 1799.0, 1802.0, 2500.0, 1996.0,
        1725.0, 1992.0, 1659.0, 1378.0, 1460.0, 1329.0, 1211.0, 1264.0,
        1263.0, 738.6, 472.1, 229.5, 129.8, 60.64, 123.4, 93.58, 83.46,
        106.9, 102.4, 97.69, 111.6, 70.6, 26.17, 12.55, 7.004, 4.32,
        1.999, 1.434, 2.073, 1.951, 1.071, 0.6014, 0.2532, 0.1414,
        0.09411, 0.07017, 0.04758, 0.0373, 0.03039, 0.02678, 0.02389,
        0.02347, 0.02475, 0.02626, 0.02772, 0.0302, 0.03213, 0.03514,
        0.03632,
      }
    },


    // Z = 84
    {
      57, {
        0.001, 0.0015, 0.002, 0.002683, 0.002683, 0.0027399, 0.002798,
        0.002798, 0.003, 0.0033019, 0.0033019, 0.00356733, 0.0038541,
        0.0038541, 0.004, 0.0041494, 0.0041494, 0.005, 0.006, 0.008,
        0.01, 0.0138138, 0.0138138, 0.015, 0.0162443, 0.0162443,
        0.0165882, 0.0169393, 0.0169393, 0.02, 0.03, 0.04, 0.05, 0.06,
        0.08, 0.093105, 0.093105, 0.1, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6,
        0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0,
        20.0,
      }, {
        5724.0, 2604.0, 1423.0, 748.2, 2206.0, 1941.0, 1775.0, 2711.0, 2155.0,
        1693.0, 1956.0, 1622.0, 1339.0, 1419.0, 1299.0, 1190.0, 1242.0,
        793.1, 508.5, 249.4, 142.7, 63.37, 151.3, 121.9, 98.49, 136.1,
        129.7, 123.2, 142.0, 93.52, 33.03, 15.69, 8.802, 5.499, 2.649,
        1.82, 7.14, 5.991, 2.17, 1.078, 0.4335, 0.2483, 0.1714, 0.1318,
        0.09286, 0.07391, 0.06087, 0.05394, 0.04749, 0.04362, 0.04323,
        0.04399, 0.04522, 0.04816, 0.05124, 0.05835, 0.06398,
      }, {
        5710.0, 2591.0, 1411.0, 736.9, 2142.0, 1884.0, 1726.0, 2625.0, 2090.0,
        1645.0, 1900.0, 1576.0, 1303.0, 1381.0, 1264.0, 1158.0, 1209.0,
        772.1, 494.2, 240.6, 136.3, 59.1, 118.9, 97.28, 79.65, 101.9,
        97.55, 93.08, 106.3, 72.78, 27.21, 13.11, 7.334, 4.532, 2.103,
        1.4, 2.005, 1.936, 1.095, 0.6207, 0.263, 0.1471, 0.09781,
        0.07281, 0.04916, 0.03839, 0.03117, 0.0274, 0.02438, 0.02392,
        0.02521, 0.02674, 0.02822, 0.03074, 0.03271, 0.03577, 0.03694,
      }
    },


    // Z = 85
    {
      60, {
        0.001, 0.00102078, 0.001042, 0.001042, 0.0015, 0.002, 0.0027867,
        0.0027867, 0.00284705, 0.0029087, 0.0029087, 0.003, 0.003426,
        0.003426, 0.004, 0.004008, 0.004008, 0.00415963, 0.004317,
        0.004317, 0.005, 0.006, 0.008, 0.01, 0.0142135, 0.0142135,
        0.015, 0.0167847, 0.0167847, 0.0171352, 0.017493, 0.017493,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.0957299, 0.0957299, 0.1,
        0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0,
        4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        5868.0, 5651.0, 5435.0, 5551.0, 2731.0, 1495.0, 722.5, 2431.0, 1928.0,
        1686.0, 2763.0, 2250.0, 1614.0, 1866.0, 1275.0, 1269.0, 1345.0,
        1234.0, 1126.0, 1175.0, 825.1, 529.9, 260.1, 149.0, 61.62,
        146.1, 127.9, 94.17, 130.2, 133.8, 118.1, 136.0, 97.04, 34.42,
        16.38, 9.196, 5.748, 2.769, 1.777, 6.895, 6.174, 2.249, 1.118,
        0.4491, 0.2565, 0.1765, 0.1354, 0.09495, 0.07538, 0.06189,
        0.05479, 0.04821, 0.04424, 0.04383, 0.04461, 0.04585, 0.04883,
        0.05195, 0.05918, 0.06493,
      }, {
        5854.0, 5637.0, 5420.0, 5537.0, 2717.0, 1482.0, 711.2, 2355.0, 1869.0,
        1638.0, 2671.0, 2178.0, 1566.0, 1811.0, 1240.0, 1233.0, 1308.0,
        1196.0, 1095.0, 1142.0, 802.5, 514.6, 250.9, 142.3, 57.38,
        114.0, 100.8, 75.76, 96.7, 92.64, 88.47, 101.0, 74.46, 28.12,
        13.61, 7.637, 4.729, 2.2, 1.361, 1.935, 1.897, 1.112, 0.6369,
        0.2718, 0.1522, 0.1012, 0.07517, 0.05055, 0.03934, 0.03181,
        0.02791, 0.02478, 0.02427, 0.02556, 0.0271, 0.02859, 0.03113,
        0.03311, 0.03619, 0.03739,
      }
    },


    // Z = 86
    {
      60, {
        0.001, 0.00104738, 0.001097, 0.001097, 0.0015, 0.002, 0.0028924,
        0.0028924, 0.003, 0.0030215, 0.0030215, 0.00326957, 0.003538,
        0.003538, 0.004, 0.004159, 0.004159, 0.00431748, 0.004482,
        0.004482, 0.005, 0.006, 0.008, 0.01, 0.0146194, 0.0146194,
        0.015, 0.0173371, 0.0173371, 0.0176895, 0.018049, 0.018049,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.098404, 0.098404, 0.1,
        0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5, 2.0, 3.0,
        4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        5826.0, 5354.0, 4909.0, 5013.0, 2719.0, 1490.0, 663.4, 2319.0, 1554.0,
        1524.0, 2470.0, 1867.0, 1479.0, 1710.0, 1266.0, 1148.0, 1217.0,
        1112.0, 1017.0, 1061.0, 816.3, 524.0, 257.7, 147.7, 56.97,
        134.4, 125.9, 85.57, 118.5, 122.5, 107.7, 124.1, 95.63, 34.08,
        16.24, 9.125, 5.706, 2.749, 1.649, 6.371, 6.086, 2.215, 1.101,
        0.442, 0.2518, 0.1728, 0.1322, 0.0923, 0.07303, 0.05984, 0.0529,
        0.04649, 0.04264, 0.04224, 0.04298, 0.04418, 0.04705, 0.05007,
        0.05704, 0.0626,
      }, {
        5812.0, 5340.0, 4895.0, 5000.0, 2706.0, 1478.0, 652.6, 2243.0, 1507.0,
        1479.0, 2384.0, 1804.0, 1433.0, 1657.0, 1229.0, 1115.0, 1182.0,
        1080.0, 987.7, 1031.0, 793.1, 508.5, 248.5, 141.1, 52.96, 104.2,
        98.07, 68.51, 87.24, 83.66, 80.02, 91.3, 72.31, 27.6, 13.42,
        7.552, 4.686, 2.185, 1.258, 1.779, 1.766, 1.073, 0.6204, 0.2668,
        0.1496, 0.09941, 0.07377, 0.04942, 0.03832, 0.03089, 0.02704,
        0.02396, 0.02344, 0.02469, 0.0262, 0.02766, 0.03016, 0.03216,
        0.03534, 0.0367,
      }
    },


    // Z = 87
    {
      61, {
        0.001, 0.00107378, 0.001153, 0.001153, 0.0015, 0.002, 0.0029999,
        0.0029999, 0.003, 0.0031362, 0.0031362, 0.00338938, 0.003663,
        0.003663, 0.004, 0.004327, 0.004327, 0.00448656, 0.004652,
        0.004652, 0.005, 0.006, 0.008, 0.01, 0.015, 0.0150312,
        0.0150312, 0.016406, 0.0179065, 0.0179065, 0.0182691, 0.018639,
        0.018639, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1, 0.101137,
        0.101137, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0, 1.25, 1.5,
        2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        6238.0, 5356.0, 4673.0, 4772.0, 2846.0, 1562.0, 641.4, 1995.0, 1994.0,
        1449.0, 2255.0, 1761.0, 1415.0, 1641.0, 1322.0, 1086.0, 1153.0,
        1059.0, 966.6, 1008.0, 848.9, 545.1, 268.4, 153.9, 55.73, 55.44,
        130.4, 103.1, 81.84, 113.7, 117.9, 103.1, 118.7, 99.33, 35.5,
        16.94, 9.525, 5.959, 2.871, 1.655, 1.61, 6.089, 2.295, 1.141,
        0.4577, 0.2601, 0.178, 0.1358, 0.09444, 0.0745, 0.06087,
        0.05376, 0.04721, 0.04324, 0.04283, 0.04358, 0.04479, 0.04769,
        0.05077, 0.05786, 0.0635,
      }, {
        6224.0, 5342.0, 4659.0, 4758.0, 2833.0, 1549.0, 630.6, 1929.0, 1928.0,
        1405.0, 2175.0, 1700.0, 1370.0, 1588.0, 1281.0, 1053.0, 1118.0,
        1023.0, 937.7, 978.0, 823.8, 528.5, 258.8, 147.1, 51.74, 51.46,
        100.3, 80.86, 65.21, 82.99, 79.47, 76.02, 86.66, 73.95, 28.49,
        13.91, 7.851, 4.881, 2.283, 1.261, 1.224, 1.71, 1.087, 0.6351,
        0.275, 0.1542, 0.1025, 0.07597, 0.05067, 0.0392, 0.03151,
        0.02752, 0.02432, 0.02372, 0.02495, 0.02644, 0.02788, 0.03033,
        0.03227, 0.03525, 0.03641,
      }
    },


    // Z = 88
    {
      65, {
        0.001, 0.0010284, 0.0010576, 0.0010576, 0.00113049, 0.0012084,
        0.0012084, 0.0015, 0.002, 0.003, 0.0031049, 0.0031049,
        0.00317584, 0.0032484, 0.0032484, 0.0035096, 0.0037918,
        0.0037918, 0.004, 0.0044895, 0.0044895, 0.00465278, 0.004822,
        0.004822, 0.005, 0.006, 0.008, 0.01, 0.015, 0.0154444,
        0.0154444, 0.0168961, 0.0184843, 0.0184843, 0.0188568,
        0.0192367, 0.0192367, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1,
        0.103922, 0.103922, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        6201.0, 5895.0, 5602.0, 5683.0, 5019.0, 4424.0, 4516.0, 2950.0,
        1620.0, 666.4, 616.9, 1715.0, 1506.0, 1374.0, 2039.0, 1647.0,
        1342.0, 1558.0, 1367.0, 1024.0, 1086.0, 999.5, 912.9, 952.2,
        874.1, 561.3, 276.9, 158.9, 57.6, 53.55, 125.3, 98.39, 77.66,
        108.1, 112.6, 97.99, 112.8, 102.3, 36.64, 17.5, 9.85, 6.166,
        2.971, 1.712, 1.56, 5.826, 2.355, 1.172, 0.4696, 0.2662, 0.1817,
        0.1383, 0.09579, 0.07535, 0.06141, 0.05415, 0.04751, 0.04348,
        0.04304, 0.04379, 0.045, 0.04793, 0.05102, 0.05815, 0.06383,
      }, {
        6187.0, 5881.0, 5588.0, 5669.0, 5005.0, 4411.0, 4502.0, 2937.0,
        1608.0, 655.5, 606.1, 1657.0, 1457.0, 1331.0, 1965.0, 1587.0,
        1297.0, 1506.0, 1322.0, 991.7, 1052.0, 964.3, 884.6, 922.8,
        847.2, 543.8, 266.9, 151.9, 53.53, 49.62, 95.81, 76.77, 61.6,
        78.35, 75.0, 71.75, 81.72, 74.96, 29.13, 14.29, 8.088, 5.039,
        2.363, 1.308, 1.181, 1.638, 1.091, 0.644, 0.2811, 0.158, 0.105,
        0.07772, 0.05165, 0.03983, 0.03191, 0.02779, 0.02452, 0.02387,
        0.02509, 0.02658, 0.02802, 0.0305, 0.03245, 0.03549, 0.03668,
      }
    },


    // Z = 89
    {
      64, {
        0.001, 0.00103923, 0.00108, 0.00108, 0.00117069, 0.001269, 0.001269,
        0.0015, 0.002, 0.003, 0.003219, 0.003219, 0.00329373, 0.0033702,
        0.0033702, 0.00362962, 0.003909, 0.003909, 0.004, 0.004656,
        0.004656, 0.005, 0.005002, 0.005002, 0.006, 0.008, 0.01, 0.015,
        0.015871, 0.015871, 0.0174031, 0.0190832, 0.0190832, 0.0194579,
        0.01984, 0.01984, 0.02, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1,
        0.106756, 0.106756, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        6469.0, 6034.0, 5622.0, 5702.0, 4905.0, 4203.0, 4289.0, 3082.0,
        1696.0, 698.1, 595.9, 1537.0, 1404.0, 1307.0, 1892.0, 1564.0,
        1296.0, 1504.0, 1426.0, 974.5, 1034.0, 868.7, 867.8, 905.1,
        582.9, 287.8, 165.3, 60.02, 52.13, 121.4, 104.2, 74.26, 103.6,
        107.9, 94.03, 108.2, 106.2, 38.11, 18.23, 10.27, 6.433, 3.1,
        1.786, 1.525, 5.622, 2.434, 1.213, 0.4859, 0.2749, 0.1872,
        0.1421, 0.09804, 0.07686, 0.06251, 0.05504, 0.04822, 0.04409,
        0.04363, 0.04438, 0.0456, 0.04856, 0.05171, 0.05895, 0.06473,
      }, {
        6455.0, 6020.0, 5608.0, 5688.0, 4891.0, 4190.0, 4275.0, 3069.0,
        1683.0, 687.0, 585.1, 1485.0, 1358.0, 1264.0, 1822.0, 1508.0,
        1252.0, 1453.0, 1377.0, 942.8, 1000.0, 840.8, 839.9, 876.1,
        564.1, 277.3, 158.0, 55.83, 48.23, 92.24, 73.5, 58.65, 74.48,
        71.33, 68.31, 77.74, 76.45, 30.0, 14.79, 8.396, 5.242, 2.465,
        1.367, 1.15, 1.582, 1.1, 0.6581, 0.2896, 0.1632, 0.1085,
        0.08018, 0.05311, 0.0408, 0.03258, 0.02831, 0.0249, 0.02419,
        0.02539, 0.02688, 0.02832, 0.03079, 0.03275, 0.03575, 0.03692,
      }
    },


    // Z = 90
    {
      63, {
        0.001, 0.00108088, 0.0011683, 0.0011683, 0.0012463, 0.0013295,
        0.0013295, 0.0015, 0.002, 0.003, 0.003332, 0.003332, 0.00341048,
        0.0034908, 0.0034908, 0.004, 0.0040461, 0.0040461, 0.00442089,
        0.0048304, 0.0048304, 0.005, 0.0051823, 0.0051823, 0.006, 0.008,
        0.01, 0.015, 0.0163003, 0.0163003, 0.0179166, 0.0196932,
        0.0196932, 0.02, 0.0204721, 0.0204721, 0.03, 0.04, 0.05, 0.06,
        0.08, 0.1, 0.109651, 0.109651, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6,
        0.8, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0,
        20.0,
      }, {
        6614.0, 5747.0, 4976.0, 5048.0, 4463.0, 3937.0, 4017.0, 3161.0,
        1742.0, 718.0, 567.1, 1394.0, 1304.0, 1226.0, 1749.0, 1253.0,
        1217.0, 1415.0, 1137.0, 910.3, 965.5, 887.8, 812.9, 847.7,
        594.5, 293.9, 168.9, 61.41, 49.91, 115.6, 99.24, 69.84, 95.27,
        93.68, 88.51, 101.8, 38.92, 18.65, 10.52, 6.592, 3.178, 1.83,
        1.465, 5.336, 2.472, 1.234, 0.4939, 0.2789, 0.1895, 0.1435,
        0.09861, 0.07709, 0.06251, 0.05498, 0.04812, 0.04396, 0.04347,
        0.04421, 0.04542, 0.04836, 0.05149, 0.05871, 0.06447,
      }, {
        6600.0, 5733.0, 4962.0, 5034.0, 4449.0, 3924.0, 4003.0, 3148.0,
        1729.0, 707.0, 556.5, 1346.0, 1260.0, 1185.0, 1683.0, 1208.0,
        1174.0, 1364.0, 1098.0, 879.7, 933.0, 858.1, 785.9, 819.6,
        574.9, 283.0, 161.5, 57.18, 46.11, 87.32, 69.24, 54.92, 68.36,
        67.14, 63.83, 72.59, 30.32, 15.03, 8.558, 5.355, 2.526, 1.403,
        1.1, 1.502, 1.09, 0.6603, 0.2931, 0.1656, 0.1101, 0.08128,
        0.05366, 0.04109, 0.03269, 0.02834, 0.02487, 0.02411, 0.02528,
        0.02674, 0.02817, 0.0306, 0.03253, 0.0355, 0.03664,
      }
    },


    // Z = 91
    {
      66, {
        0.001, 0.00100334, 0.0010067, 0.0010067, 0.00111018, 0.0012243,
        0.0012243, 0.00130316, 0.0013871, 0.0013871, 0.0015, 0.002,
        0.003, 0.0034418, 0.0034418, 0.00352548, 0.0036112, 0.0036112,
        0.004, 0.0041738, 0.0041738, 0.005, 0.0050009, 0.0050009,
        0.00518067, 0.0053669, 0.0053669, 0.006, 0.008, 0.01, 0.015,
        0.0167331, 0.0167331, 0.02, 0.0203137, 0.0203137, 0.0207054,
        0.0211046, 0.0211046, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1,
        0.112601, 0.112601, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        6530.0, 6490.0, 6451.0, 6868.0, 5755.0, 4794.0, 4863.0, 4311.0,
        3815.0, 3891.0, 3327.0, 1834.0, 755.8, 554.8, 1363.0, 1266.0,
        1185.0, 1695.0, 1315.0, 1182.0, 1374.0, 875.9, 875.5, 928.8,
        851.4, 781.0, 814.4, 621.7, 307.4, 176.9, 64.42, 49.03, 113.0,
        70.25, 67.47, 94.52, 89.95, 85.76, 98.61, 40.77, 19.57, 11.05,
        6.929, 3.342, 1.924, 1.445, 5.198, 2.575, 1.288, 0.5152, 0.2904,
        0.1969, 0.1487, 0.1018, 0.07937, 0.0642, 0.05637, 0.04927,
        0.04496, 0.04445, 0.04519, 0.04641, 0.04943, 0.05261, 0.06,
        0.06592,
      }, {
        6515.0, 6476.0, 6437.0, 6853.0, 5740.0, 4780.0, 4848.0, 4296.0,
        3801.0, 3877.0, 3313.0, 1821.0, 744.4, 544.0, 1314.0, 1222.0,
        1144.0, 1628.0, 1265.0, 1139.0, 1324.0, 845.4, 845.0, 896.4,
        821.9, 754.1, 786.4, 600.4, 295.8, 169.0, 60.0, 45.22, 84.83,
        54.86, 52.84, 67.04, 64.18, 61.53, 69.9, 31.46, 15.67, 8.95,
        5.612, 2.655, 1.477, 1.081, 1.466, 1.106, 0.6792, 0.3042,
        0.1723, 0.1146, 0.08453, 0.05563, 0.04247, 0.03367, 0.02911,
        0.02547, 0.02463, 0.0258, 0.02727, 0.0287, 0.03117, 0.03311,
        0.03609, 0.03724,
      }
    },


    // Z = 92
    {
      66, {
        0.001, 0.0010222, 0.0010449, 0.0010449, 0.00115314, 0.0012726,
        0.0012726, 0.00135409, 0.0014408, 0.0014408, 0.0015, 0.002,
        0.003, 0.0035517, 0.0035517, 0.00363859, 0.0037276, 0.0037276,
        0.004, 0.0043034, 0.0043034, 0.005, 0.0051822, 0.0051822,
        0.00536198, 0.005548, 0.005548, 0.006, 0.008, 0.01, 0.015,
        0.0171663, 0.0171663, 0.02, 0.0209476, 0.0209476, 0.0213487,
        0.0217574, 0.0217574, 0.03, 0.04, 0.05, 0.06, 0.08, 0.1,
        0.115606, 0.115606, 0.15, 0.2, 0.3, 0.4, 0.5, 0.6, 0.8, 1.0,
        1.25, 1.5, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        6626.0, 6375.0, 6127.0, 6519.0, 5446.0, 4526.0, 4589.0, 4065.0,
        3598.0, 3668.0, 3382.0, 1865.0, 769.2, 525.7, 1266.0, 1188.0,
        1112.0, 1582.0, 1329.0, 1110.0, 1292.0, 889.1, 811.8, 861.1,
        791.5, 728.2, 759.2, 628.4, 310.8, 179.1, 65.28, 46.63, 107.0,
        71.06, 63.0, 88.38, 95.15, 80.23, 92.22, 41.28, 19.83, 11.21,
        7.035, 3.395, 1.954, 1.378, 4.893, 2.591, 1.298, 0.5192, 0.2922,
        0.1976, 0.149, 0.1016, 0.07896, 0.0637, 0.05587, 0.04878,
        0.04447, 0.04392, 0.04463, 0.04583, 0.04879, 0.05195, 0.05927,
        0.06512,
      }, {
        6612.0, 6360.0, 6112.0, 6504.0, 5431.0, 4511.0, 4575.0, 4051.0,
        3584.0, 3654.0, 3368.0, 1852.0, 757.8, 515.1, 1220.0, 1142.0,
        1072.0, 1517.0, 1277.0, 1068.0, 1242.0, 856.9, 782.5, 830.1,
        763.2, 702.2, 732.2, 606.2, 298.9, 171.1, 60.84, 42.93, 79.8,
        54.96, 49.14, 62.22, 59.59, 57.15, 64.87, 31.49, 15.76, 9.034,
        5.678, 2.695, 1.502, 1.027, 1.382, 1.083, 0.6746, 0.305, 0.1732,
        0.1152, 0.08494, 0.05574, 0.04241, 0.03351, 0.02891, 0.02523,
        0.02434, 0.02546, 0.02689, 0.02829, 0.03068, 0.03259, 0.03552,
        0.03662,
      }
    },


    // Z = 93 -- compound material CdTe (NOT Np)
    {
      59, {
        0.0010, 0.001003, 0.001006, 0.001006, 0.00150, 0.0020, 0.0030,
        0.003537, 0.003537, 0.003631, 0.003727, 0.003727, 0.0040, 0.004018,
        0.004018, 0.004177, 0.004341, 0.004341, 0.004475, 0.004612, 0.004612,
        0.004773, 0.004939, 0.004939, 0.0050, 0.0060, 0.0080, 0.010, 0.0150,
        0.020, 0.026711, 0.026711, 0.030, 0.031814, 0.031814, 0.040, 0.050,
        0.060, 0.080, 0.10, 0.150, 0.20, 0.30, 0.40, 0.50, 0.60, 0.80, 1.0,
        1.250, 1.50, 2.0, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 15.0, 20.0,
      }, {
        7927.0, 7875.0, 7824.0, 8014.0, 3291.0, 1664.0, 614.60, 406.40,
        778.70, 730.0, 684.0, 860.10, 723.0, 715.10, 793.40, 722.10, 656.20,
        932.80, 873.90, 813.50, 943.80, 870.20, 799.90, 865.30, 839.20,
        528.60, 249.20, 138.10, 46.570, 21.440, 9.8340, 29.430, 21.820,
        18.730, 34.920, 19.30, 10.670, 6.5420, 3.0190, 1.6710, 0.60710,
        0.32460, 0.16280, 0.11470, 0.092910, 0.080420, 0.0660, 0.057420,
        0.050430, 0.045910, 0.04070, 0.036490, 0.035250, 0.035130, 0.035480,
        0.036870, 0.038570, 0.042730, 0.046160,
      }, {
        7913.0, 7862.0, 7811.0, 8001.0, 3281.0, 1656.0, 608.20, 400.40,
        752.50, 705.60, 661.30, 826.60, 695.70, 688.10, 762.40, 694.40,
        631.30, 888.30, 832.80, 775.90, 895.50, 826.50, 760.60, 821.40,
        797.0, 504.60, 238.90, 132.20, 44.040, 19.880, 8.8330, 13.680,
        11.190, 10.010, 13.510, 9.6350, 6.2290, 4.1630, 2.0830, 1.1810,
        0.41010, 0.19820, 0.081440, 0.051090, 0.039520, 0.033910, 0.028520,
        0.025750, 0.023540, 0.022090, 0.020620, 0.020210, 0.020910, 0.021880,
        0.022860, 0.024680, 0.026180, 0.028670, 0.029830,
      }
    },

    // entry 94 -- compound material GaAs
    {
      58, {
        0.00100000,  0.00105613,  0.00111540,  0.00111540,  0.00112877,  0.00114230,
        0.00114230,  0.00121752,  0.00129770,  0.00129770,  0.00131034,  0.00132310,
        0.00132310,  0.00134073,  0.00135860,  0.00135860,  0.00150000,  0.00152650,
        0.00152650,  0.00200000,  0.00300000,  0.00400000,  0.00500000,  0.00600000,
        0.00800000,  0.01000000,  0.01036710,  0.01036710,  0.01109160,  0.01186670,
        0.01186670,  0.01500000,  0.02000000,  0.03000000,  0.04000000,  0.05000000,
        0.06000000,  0.08000000,  0.10000000,  0.15000000,  0.20000000,  0.30000000,
        0.40000000,  0.50000000,  0.60000000,  0.80000000,  1.00000000,  1.25000000,
        1.50000000,  2.00000000,  3.00000000,  4.00000000,  5.00000000,  6.00000000,
        8.00000000, 10.00000000, 15.00000000, 20.00000000,
      }, {
        1.917e+03, 1.685e+03, 1.481e+03, 2.772e+03, 3.180e+03, 3.532e+03, 4.372e+03,
        4.130e+03, 3.657e+03, 4.066e+03, 3.971e+03, 3.879e+03, 5.652e+03, 5.525e+03,
        5.415e+03, 6.266e+03, 5.159e+03, 4.939e+03, 5.278e+03, 2.731e+03, 9.702e+02,
        4.539e+02, 2.495e+02, 1.524e+02, 6.960e+01, 3.780e+01, 3.425e+01, 1.260e+02,
        1.059e+02, 8.899e+01, 1.685e+02, 9.220e+01, 4.258e+01, 1.397e+01, 6.262e+00,
        3.365e+00, 2.042e+00, 9.587e-01, 5.598e-01, 2.509e-01, 1.671e-01, 1.137e-01,
        9.371e-02, 8.248e-02, 7.484e-02, 6.452e-02, 5.751e-02, 5.122e-02, 4.676e-02,
        4.102e-02, 3.538e-02, 3.288e-02, 3.172e-02, 3.121e-02, 3.117e-02, 3.170e-02,
        3.354e-02, 3.543e-02,
      }, {
        1.911e+03, 1.680e+03, 1.476e+03, 2.752e+03, 3.156e+03, 3.504e+03, 4.333e+03,
        4.094e+03, 3.626e+03, 4.031e+03, 3.937e+03, 3.846e+03, 5.593e+03, 5.468e+03,
        5.360e+03, 6.197e+03, 5.106e+03, 4.888e+03, 5.224e+03, 2.706e+03, 9.614e+02,
        4.486e+02, 2.456e+02, 1.492e+02, 6.728e+01, 3.601e+01, 3.254e+01, 8.315e+01,
        7.188e+01, 6.193e+01, 1.016e+02, 6.257e+01, 3.189e+01, 1.132e+01, 5.187e+00,
        2.779e+00, 1.655e+00, 7.246e-01, 3.838e-01, 1.299e-01, 6.921e-02, 3.916e-02,
        3.210e-02, 2.943e-02, 2.802e-02, 2.628e-02, 2.499e-02, 2.361e-02, 2.250e-02,
        2.101e-02, 1.984e-02, 1.969e-02, 1.994e-02, 2.034e-02, 2.127e-02, 2.215e-02,
        2.370e-02, 2.459e-02,
      }
    },

  };

}}}
