/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.html.editor.completion;

import java.io.IOException;
import javax.swing.text.BadLocationException;
import junit.framework.Test;
import junit.framework.TestSuite;
import org.netbeans.modules.html.editor.lib.api.HtmlVersion;
import org.netbeans.modules.html.editor.completion.HtmlCompletionTestSupport.Match;
import org.netbeans.modules.html.parser.HtmlDocumentation;
import org.netbeans.modules.parsing.spi.ParseException;

/**
 *
 * @author marekfukala
 */
public class Html5CompletionQueryTest extends HtmlCompletionQueryTest {

    private static final String HTML5_DOCTYPE = "<!doctype html>";

    public Html5CompletionQueryTest(String name) throws IOException, BadLocationException {
        super(name);
    }

    @Override
    protected HtmlVersion getExpectedVersion() {
        return HtmlVersion.HTML5;
    }

    @Override
    protected void setUp() throws Exception {
        super.setUp();
        HtmlVersion.DEFAULT_VERSION_UNIT_TESTS_OVERRIDE = HtmlVersion.HTML5;
        HtmlDocumentation.setupDocumentationForUnitTests();
    }

    public static Test Xsuite() throws IOException, BadLocationException {
        System.err.println("Beware, only selected tests runs!!!");
        TestSuite suite = new TestSuite();
        suite.addTest(new Html5CompletionQueryTest("testEndTagAutocompletion"));
        return suite;
    }

    @Override
    public void testTagAttributeValues() throws BadLocationException, ParseException {
        assertItems("<div contenteditable=\"|\"", arr("false"), Match.CONTAINS, 22);
        //           012345678901234567890 12 34567
        assertItems("<div contenteditable=\"tr|\"", arr("true"), Match.CONTAINS, 22);
        assertItems("<div contenteditable=\"true|\"", arr("true"), Match.EXACT, 22);

    }
    
    @Override
      public void testCompleteTagAttributeValues() throws BadLocationException, ParseException {
        assertCompletedText("<div contenteditable=\"|\"", "true", "<div contenteditable=\"true|\"");
        assertCompletedText("<div contenteditable=\"tr|\"", "true", "<div contenteditable=\"true|\"");

        //regression test - issue #161852
        assertCompletedText("<div contenteditable=\"|\"", "false", "<div contenteditable=\"false|\"");

        //test single quote
        assertCompletedText("<div contenteditable='|'", "false", "<div contenteditable='false'|");

        //test values cc without quotation
        assertCompletedText("<div contenteditable=|", "false", "<div contenteditable=false|");
        assertCompletedText("<div contenteditable=tr|", "true", "<div contenteditable=true|");
    }

    
    @Override
    public void testNoEndTagAutocompletion() {
        //disabled for html5, the end tag autocompletion works even for unknown tags
    }

    @Override
    public void testBooleanAttributes() throws BadLocationException, ParseException {
        System.err.println("testBooleanAttributes test is disabled due to a bug!");
        //disabled, fails since there's no information if an attribute is empty or not so the
        //equal sign plus quotations is generated by the completion even if the value is forbidden
    }

    public void testIssue193268() throws IOException, BadLocationException, ParseException {
        TestSource sap = getTestSource("issue193268.html", false);
        assertItems(sap.getCode(), new String[]{"id", "href"} , Match.CONTAINS);
    }
    
    //Bug 197608 - Non-html tags offered as closing tags using code completion 
    public void testIssue197608() throws BadLocationException, ParseException {
        assertItems("<div></di|", arr("div"), Match.EXACT);
        assertCompletedText("<div></di|", "/div", "<div></div>|");
    }
    
    //Bug 197614 - Problem with HTML4 & code completion - non matching tags offered
    public void testIssue197614() throws BadLocationException, ParseException {
        assertItems("<table><tr><td></ta|", arr("table"), Match.EXACT);
    }

    @Override
    protected void assertItems(String documentText, final String[] expectedItemsNames, final Match type, int expectedAnchor) throws BadLocationException, ParseException {
        super.assertItems(HTML5_DOCTYPE + documentText,
                expectedItemsNames,
                type,
                expectedAnchor != -1 ? HTML5_DOCTYPE.length() + expectedAnchor : -1);
    }
}
