/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.streampipes.extensions.management.monitoring;

import org.apache.streampipes.extensions.api.monitoring.IExtensionsLogger;
import org.apache.streampipes.extensions.api.monitoring.SpMonitoringManager;
import org.apache.streampipes.model.monitoring.SpLogEntry;
import org.apache.streampipes.model.monitoring.SpLogMessage;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class ExtensionsLogger implements IExtensionsLogger {

  private static final Logger LOG = LoggerFactory.getLogger(ExtensionsLogger.class);

  private final String resourceId;
  private final SpMonitoringManager monitoringManager;

  public ExtensionsLogger(String resourceId) {
    this.resourceId = resourceId;
    this.monitoringManager = SpMonitoringManager.INSTANCE;
  }

  @Override
  public void log(SpLogMessage logMessage) {
    monitoringManager.addErrorMessage(
        resourceId,
        SpLogEntry.from(System.currentTimeMillis(), logMessage));
  }

  @Override
  public void error(Exception e) {
    LOG.error("Error while processing the internal pipeline", e);
    log(SpLogMessage.from(e));
  }

  @Override
  public void error(String details, Exception e) {
    LOG.error("Error while processing the internal pipeline", e);
    log(SpLogMessage.from(e, details));
  }

  @Override
  public void info(String title, String details) {
    LOG.info("Info message while processing the internal pipeline: {}, {}", title, details);
    log(SpLogMessage.info(title, details));
  }

  @Override
  public void warn(String title, String details) {
    LOG.warn("Warning while processing the internal pipeline: {}, {}", title, details);
    log(SpLogMessage.warn(title, details));
  }

}
