/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

import { Component, OnInit, ViewChild, inject } from '@angular/core';
import { ValidatorFn, Validators } from '@angular/forms';
import { ConfigurationInfo } from '../../../connect/model/ConfigurationInfo';
import {
    DataType,
    FreeTextStaticProperty,
    SemanticType,
} from '@streampipes/platform-services';
import {
    ValidateNumber,
    ValidateString,
    ValidateUrl,
} from '../input.validator';
import { AbstractValidatedStaticPropertyRenderer } from '../base/abstract-validated-static-property';
import { QuillEditorComponent } from 'ngx-quill';
import { TranslateService } from '@ngx-translate/core';
import { CdkTextareaAutosize } from '@angular/cdk/text-field';

@Component({
    selector: 'sp-app-static-free-input',
    templateUrl: './static-free-input.component.html',
    styleUrls: ['./static-free-input.component.scss'],
    standalone: false,
})
export class StaticFreeInputComponent
    extends AbstractValidatedStaticPropertyRenderer<FreeTextStaticProperty>
    implements OnInit
{
    translateService = inject(TranslateService);
    quillModules: any = {
        toolbar: [
            ['bold', 'italic', 'underline', 'strike'],
            [{ header: 1 }, { header: 2 }],
            [{ size: ['small', false, 'large', 'huge'] }],
            [{ header: [1, 2, 3, 4, 5, 6, false] }],
            [{ color: [] }, { background: [] }],
        ],
    };

    quillModulesFontFormat: any = {
        toolbar: [['bold', 'italic', 'underline', 'strike']],
    };

    @ViewChild('textEditor', { static: false })
    quillEditorComponent: QuillEditorComponent;

    @ViewChild('autosize') autosize: CdkTextareaAutosize;

    constructor() {
        super();
    }

    ngOnInit() {
        this.addValidator(this.staticProperty.value, this.collectValidators());
        this.enableValidators();
        this.emitUpdate();
        console.log(this.staticProperty);
    }

    collectValidators() {
        const validators: ValidatorFn[] = [];
        if (!this.staticProperty.optional) {
            validators.push(Validators.required);
        }
        if (
            DataType.isNumberType(this.staticProperty.requiredDatatype) ||
            DataType.isNumberType(this.staticProperty.requiredDomainProperty)
        ) {
            validators.push(ValidateNumber);
            this.errorMessage = this.translateService.instant(
                'The value should be a number',
            );
        } else if (
            this.staticProperty.requiredDomainProperty === SemanticType.SO_URL
        ) {
            validators.push(ValidateUrl);
            this.errorMessage = this.translateService.instant(
                'Please enter a valid URL',
            );
        } else if (this.staticProperty.requiredDatatype === DataType.STRING) {
            validators.push(ValidateString);
            this.errorMessage = this.translateService.instant(
                'Please enter a valid String',
            );
        }

        return validators;
    }

    emitUpdate() {
        const valid =
            this.staticProperty.value !== undefined &&
            this.staticProperty.value !== '' &&
            this.staticProperty.value !== null;
        this.applyCompletedConfiguration(valid);
    }

    onStatusChange(status: any) {}

    onValueChange(value: any) {
        this.staticProperty.value = value;
        this.parentForm.updateValueAndValidity();
    }

    applyPlaceholder(runtimeName) {
        const valueToInsert = '#' + runtimeName + '#';
        if (this.quillEditorComponent) {
            const currentIndex =
                this.quillEditorComponent.quillEditor.selection.savedRange
                    .index;
            this.quillEditorComponent.quillEditor.insertText(
                currentIndex,
                valueToInsert,
                'user',
            );
        } else {
            this.parentForm.controls[this.fieldName].setValue(
                this.parentForm.controls[this.fieldName].value +
                    ' ' +
                    valueToInsert,
            );
        }
    }

    formatLabel(value: number) {
        if (!Number.isInteger(value)) {
            value = Number(value.toFixed(1));
        }
        return value;
    }
}
