package linode

import (
	"github.com/caddyserver/caddy/v2"
	"github.com/caddyserver/caddy/v2/caddyconfig/caddyfile"
	"github.com/libdns/linode"
)

// Provider lets Caddy read and manipulate DNS records hosted by this DNS provider.
type Provider struct{ *linode.Provider }

func init() {
	caddy.RegisterModule(Provider{})
}

// CaddyModule returns the Caddy module information.
func (Provider) CaddyModule() caddy.ModuleInfo {
	return caddy.ModuleInfo{
		ID:  "dns.providers.linode",
		New: func() caddy.Module { return &Provider{new(linode.Provider)} },
	}
}

// Provision sets up the module. Implements caddy.Provisioner.
func (p *Provider) Provision(_ caddy.Context) error {
	repl := caddy.NewReplacer()
	p.Provider.APIToken = repl.ReplaceAll(p.Provider.APIToken, "")
	p.Provider.APIURL = repl.ReplaceAll(p.Provider.APIURL, "")
	p.Provider.APIVersion = repl.ReplaceAll(p.Provider.APIVersion, "")
	return nil
}

// UnmarshalCaddyfile sets up the DNS provider from Caddyfile tokens. Syntax:
//
//	linode <api_token>
//
// or, when specifying optional fields:
//
//	linode [<api_token>] {
//	    api_token <string>
//	    api_url <string>
//	    api_version <string>
//	    debug_logs_enabled <true|false>
//	}
func (p *Provider) UnmarshalCaddyfile(d *caddyfile.Dispenser) error {
	for d.Next() {
		if d.NextArg() {
			p.Provider.APIToken = d.Val()
		}
		if d.NextArg() {
			return d.ArgErr()
		}
		for nesting := d.Nesting(); d.NextBlock(nesting); {
			switch d.Val() {
			case "api_token":
				if p.Provider.APIToken != "" {
					return d.Err("API token already set")
				}
				if d.NextArg() {
					p.Provider.APIToken = d.Val()
				}
				if d.NextArg() {
					return d.ArgErr()
				}
			case "api_url":
				if p.Provider.APIURL != "" {
					return d.Err("API URL already set")
				}
				if d.NextArg() {
					p.Provider.APIURL = d.Val()
				}
				if d.NextArg() {
					return d.ArgErr()
				}
			case "api_version":
				if p.Provider.APIVersion != "" {
					return d.Err("API version already set")
				}
				if d.NextArg() {
					p.Provider.APIVersion = d.Val()
				}
				if d.NextArg() {
					return d.ArgErr()
				}
			case "debug_logs_enabled":
				if d.NextArg() {
					enableString := d.Val()
					if enableString == "true" {
						p.Provider.DebugLogsEnabled = true
					} else if enableString == "false" {
						p.Provider.DebugLogsEnabled = false
					} else {
						return d.Errf("invalid value for debug_logs_enabled: %s. Should be true or false", enableString)
					}
				}
				if d.NextArg() {
					return d.ArgErr()
				}
			default:
				return d.Errf("unrecognized subdirective '%s'", d.Val())
			}
		}
	}
	if p.Provider.APIToken == "" {
		return d.Err("missing API token")
	}
	return nil
}

// Interface guards
var (
	_ caddyfile.Unmarshaler = (*Provider)(nil)
	_ caddy.Provisioner     = (*Provider)(nil)
)
