/*
 * SPDX-FileCopyrightText: 2020 Han Young <hanyoung@protonmail.com>
 * SPDX-FileCopyrightText: 2022 by Devin Lin <devin@kde.org>
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "flashlightutil.h"

#include <cstring>
#include <fcntl.h>
#include <libudev.h>
#include <unistd.h>

#include <QDebug>
#include <QFileInfo>

#define TORCH_SUBSYSTEM "leds"

FlashlightUtil::FlashlightUtil(QObject *parent)
    : QObject{parent}
    , m_device{nullptr}
    , m_isAvailable{false}
{
    findTorchDevice();
}

FlashlightUtil::~FlashlightUtil()
{
    if (m_device != nullptr) {
        udev_device_unref(m_device);
    }
}

void FlashlightUtil::toggleTorch()
{
    if (!isAvailable()) {
        qWarning() << "Flashlight not available";
        return;
    }

    int ret = udev_device_set_sysattr_value(m_device, "brightness", const_cast<char *>(m_torchEnabled ? "0" : m_maxBrightness));
    if (ret < 0) {
        qWarning() << "Flashlight can't be toggled";
        return;
    }

    m_torchEnabled = !m_torchEnabled;
    Q_EMIT torchChanged(m_torchEnabled);
}

bool FlashlightUtil::torchEnabled() const
{
    return m_torchEnabled;
}

bool FlashlightUtil::isAvailable() const
{
    return m_isAvailable;
}

void FlashlightUtil::findTorchDevice()
{
    if (m_device != nullptr) {
        udev_device_unref(m_device);
    }
    m_device = nullptr;
    m_isAvailable = false;

    struct udev *udev = udev_new();
    struct udev_enumerate *enumerate = udev_enumerate_new(udev);

    udev_enumerate_add_match_subsystem(enumerate, TORCH_SUBSYSTEM);
    udev_enumerate_add_match_sysname(enumerate, "*:torch");
    udev_enumerate_add_match_sysname(enumerate, "*:flash");
    udev_enumerate_scan_devices(enumerate);

    struct udev_list_entry *devices = udev_enumerate_get_list_entry(enumerate);
    struct udev_list_entry *entry = udev_list_entry_get_next(devices);

    if (entry == nullptr) {
        qWarning() << "No flashlight found";
        return;
    }

    const char *path = udev_list_entry_get_name(entry);

    if (path == nullptr) {
        qWarning() << "Failed to get path from udev entry";
        return;
    }

    struct udev_device *device = udev_device_new_from_syspath(udev, path);

    if (device == nullptr) {
        qWarning() << "Failed to get udev device";
        return;
    }

    const char *maxBrightness = udev_device_get_sysattr_value(device, "max_brightness");

    if (maxBrightness == nullptr) {
        qWarning() << "Failed to read max_brightness from udev device";
        return;
    }

    const char *brightness = udev_device_get_sysattr_value(device, "brightness");

    if (brightness == nullptr) {
        qWarning() << "Failed to read brightness from udev device";
        return;
    }

    m_maxBrightness = maxBrightness;
    m_device = device;
    m_isAvailable = true;
    m_torchEnabled = std::strcmp(brightness, "0") != 0;

    udev_enumerate_unref(enumerate);
    udev_unref(udev);
}