/**
 * SPDX-FileCopyrightText: 2020 Nicolas Fella <nicolas.fella@gmx.de>
 * SPDX-FileCopyrightText: 2022 ivan tkachenko <me@ratijas.tk>
 * SPDX-FileCopyrightText: 2024 Shubham Arora <shubhamarora@protonmail.com>
 *
 * SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only OR LicenseRef-KDE-Accepted-GPL
 */

pragma ComponentBehavior: Bound

import QtQuick
import QtQuick.Layouts
import QtQuick.Controls as QQC2

import org.kde.kirigami as Kirigami
import org.kde.kcmutils as KCMUtils

import org.kde.bluezqt as BluezQt
import org.kde.bluedevil.components as BluedevilComponents

import "script.js" as Script

KCMUtils.SimpleKCM {
    id: root

    required property BluezQt.Device device

    title: device.name

    Connections {
        target: root.device
        function onDeviceRemoved(device: BluezQt.Device): void {
            root.pop();
        }
    }

    Connections {
        target: root.device.adapter
        function onAdapterRemoved(adapter: BluezQt.Adapter): void {
            root.pop();
        }
        function onPoweredChanged(powered: bool): void {
            if (!powered) {
                root.pop();
            }
        }
    }

    function pop(): void {
        root.KCMUtils.ConfigModule.pop();
    }

    Connections {
        target: root.KCMUtils.ConfigModule
        function onNetworkAvailable(service: string, available: bool): void {
            switch (service) {
            case "dun":
                dunButton.visible = available && root.device.connected;
                break;
            case "nap":
                napButton.visible = available && root.device.connected;
                break;
            default:
                break;
            }
        }
    }

    Connections {
        target: root.device

        function onConnectedChanged(connected: bool): void {
            root.checkNetworkConnection();
        }
    }

    Component.onCompleted: {
        checkNetworkConnection();
    }

    function checkNetworkConnection(): void {
        KCMUtils.ConfigModule.checkNetworkConnection(device.uuids, device.address);
    }

    function makeCall(call: BluezQt.PendingCall): void {
        indicator.running = true
        call.finished.connect(call => {
            indicator.running = false
            if (call.error) {
                errorMessage.text = call.errorText
                errorMessage.visible = true
            }
        })
    }

    BluedevilComponents.ForgetDeviceDialog {
        id: forgetDeviceDialog

        parent: root.QQC2.Overlay.overlay

        onCall: call => {
            root.makeCall(call);
        }
    }

    headerPaddingEnabled: false // Let the InlineMessage touch the edges
    header: Kirigami.InlineMessage {
        id: errorMessage
        position: Kirigami.InlineMessage.Position.Header
        type: Kirigami.MessageType.Error
        showCloseButton: true
    }

    ColumnLayout {
        spacing: Kirigami.Units.smallSpacing

        Kirigami.Icon {
            source: root.device.icon
            Layout.preferredWidth: Kirigami.Units.iconSizes.enormous
            Layout.preferredHeight: Layout.preferredWidth
            Layout.alignment: Qt.AlignHCenter
        }

        Kirigami.FormLayout {

            Row {
                spacing: Kirigami.Units.smallSpacing

                QQC2.Button {
                    id: connectButton
                    enabled: !indicator.running && !root.device.blocked
                    text: root.device.connected ? i18n("Disconnect") : i18n("Connect")
                    icon.name: root.device.connected ? "network-disconnect-symbolic" : "network-connect-symbolic"

                    QQC2.ToolTip.text: i18n("Unblock the device first to be able to connect to it")
                    QQC2.ToolTip.visible: root.device.blocked && hovered

                    onClicked: {
                        if (root.device.connected) {
                            root.makeCall(root.device.disconnectFromDevice())
                        } else {
                            root.makeCall(root.device.connectToDevice())
                        }
                    }
                }

                QQC2.BusyIndicator {
                    id: indicator
                    running: false
                    height: connectButton.height
                }
            }

            QQC2.Label {
                text: Script.deviceTypeToString(root.device)
                Kirigami.FormData.label: i18n("Type:")
            }

            QQC2.Label {
                text: root.device.battery !== null ? i18n("%1%", root.device.battery.percentage) : ""
                visible: root.device.battery !== null
                Kirigami.FormData.label: i18n("Battery:")
            }

            QQC2.Label {
                text: root.device.address
                Kirigami.FormData.label: i18n("Address:")
            }

            QQC2.Label {
                text: root.device.adapter.name
                Kirigami.FormData.label: i18n("Adapter:")
            }

            QQC2.TextField {
                text: root.device.name
                onTextEdited: root.device.name = text
                Kirigami.FormData.label: i18n("Name:")
            }

            QQC2.Switch {
                text: i18n("Trusted")
                checked: root.device.trusted
                onClicked: root.device.trusted = !root.device.trusted
            }

            RowLayout {
                spacing: Kirigami.Units.smallSpacing

                QQC2.Switch {
                    text: i18n("Blocked")
                    checked: root.device.blocked
                    onClicked: root.device.blocked = !root.device.blocked
                }

                Kirigami.ContextualHelpButton {
                    toolTipText: i18n("When enabled, all incoming connections from this device will be rejected.")
                }
            }

            QQC2.Button {
                icon.name: "document-share-symbolic"
                text: i18n("Send File")
                visible: root.device.uuids.includes(BluezQt.Services.ObexObjectPush) && root.device.connected
                onClicked: root.KCMUtils.ConfigModule.runSendFile(root.device.ubi)
            }

            QQC2.Button {
                id: napButton
                icon.name: "network-wireless-bluetooth-symbolic"
                text: i18n("Setup NAP Network…")
                visible: false
                onClicked: root.KCMUtils.ConfigModule.setupNetworkConnection("nap", root.device.address, root.device.name)
            }

            QQC2.Button {
                id: dunButton
                icon.name: "network-wireless-bluetooth-symbolic"
                text: i18n("Setup DUN Network…")
                visible: false
                onClicked: root.KCMUtils.ConfigModule.setupNetworkConnection("dun", root.device.address, root.device.name)
            }

            Kirigami.Separator {
                Kirigami.FormData.isSection: true
            }

            QQC2.Button {
                action: ForgetDeviceAction {
                    dialog: forgetDeviceDialog
                    device: root.device
                }
                // override action's text with a shorter label
                text: i18nc("@action:button Forget the Bluetooth device", "Forget");
            }
        }
    }
}
